#include "mns_client.h"
#include "mns_utils.h"
#include "mns_network_tool.h"
#include "mns_common_tool.h"
#include "constants.h"

#include <curl/curl.h>
#include <sstream>
#include <ctime>
#include <iostream>

using namespace std;
using namespace mns::sdk;

class MNSGlobalFlagInit
{
public:
    MNSGlobalFlagInit()
    {
        curl_global_init(CURL_GLOBAL_ALL);
    }
    ~MNSGlobalFlagInit()
    {
        curl_global_cleanup();
    }
};
static MNSGlobalFlagInit sMNSInit;

MNSClient::MNSClient(const string& endpoint,
                     const string& accessId,
                     const string& accessKey,
                     const int32_t connPoolSize)
    : mAccessId(accessId)
    , mAccessKey(accessKey)
{
    mMnsConnTool.reset(new MNSConnectionTool(connPoolSize));

    mEndpoint = StringTool::RightTrimString(endpoint);
    mEndpoint = StringTool::RightTrimString(mEndpoint, '/');
}

void MNSClient::sendRequest(Request& request,
                            Response& response,
                            const std::string& endpoint,
                            const std::string& accessId,
                            const std::string& accessKey,
                            MNSConnectionToolPtr mnsConnTool)
{
    MNSClient::signRequest(request, endpoint, accessId, accessKey);
    MNSNetworkTool::SendRequest(endpoint,
                                request,
                                response,
                                mnsConnTool);
}

void MNSClient::signRequest(Request& req,
                            const std::string& endpoint,
                            const std::string& accessId,
                            const std::string& accessKey)
{
    const std::string& canonicalizedResource = req.generateCanonicalizedResource();
    const std::string& requestBody = req.generateRequestBody();
    size_t contentLength = strlen(requestBody.c_str());

    size_t pos = endpoint.find_first_of("//");
    req.setHeader(HOST, endpoint.substr(pos + 2));
    req.setHeader(CONTENT_TYPE, DEFAULT_CONTENT_TYPE);
    req.setHeader(DATE, TimeTool::GetDateTime());
    req.setHeader(MNS_VERSION, CURRENT_VERSION);
    if (contentLength > 0)
        req.setHeader(CONTENT_LENGTH, StringTool::ToString(contentLength));
    req.setHeader(AUTHORIZATION,
        MNSNetworkTool::Signature(req.getMethod(), canonicalizedResource,
            accessId, accessKey, req.getHeaders()));

}

QueuePtr MNSClient::createQueue(const std::string& queueName)
{
    CreateQueueRequest req(queueName);
    CreateQueueResponse resp;
    MNSClient::sendRequest(req, resp, mEndpoint, mAccessId, mAccessKey, mMnsConnTool);

    return QueuePtr(new Queue(queueName, mEndpoint, mAccessId,
        mAccessKey, mMnsConnTool));
}

QueuePtr MNSClient::createQueue(const std::string& queueName,
                  const QueueAttributes& attributes)
{
    CreateQueueRequest req(queueName, attributes);
    CreateQueueResponse resp;
    MNSClient::sendRequest(req, resp, mEndpoint, mAccessId, mAccessKey, mMnsConnTool);

    return QueuePtr(new Queue(queueName, mEndpoint, mAccessId,
        mAccessKey, mMnsConnTool));
}

QueuePtr MNSClient::getQueueRef(const std::string& queueName)
{
    return QueuePtr(new Queue(queueName, mEndpoint, mAccessId,
        mAccessKey, mMnsConnTool));
}

void MNSClient::deleteQueue(const std::string& queueName)
{
    DeleteQueueRequest req(queueName);
    DeleteQueueResponse resp;
    MNSClient::sendRequest(req, resp, mEndpoint, mAccessId, mAccessKey, mMnsConnTool);
}

void MNSClient::listQueue(const std::string& prefix,
    std::vector<std::string>& queueNames)
{
    ListQueueRequest req(prefix);
    ListQueueResponse resp(queueNames);
    MNSClient::sendRequest(req, resp, mEndpoint, mAccessId, mAccessKey, mMnsConnTool);
}

void MNSClient::listQueue(ListQueueRequest& req,
                          std::vector<std::string>& queueNames)
{
    ListQueueResponse resp(queueNames);
    MNSClient::sendRequest(req, resp, mEndpoint, mAccessId, mAccessKey, mMnsConnTool);
}

Queue::Queue(const std::string& queueName,
      const std::string& endpoint,
      const std::string& accessId,
      const std::string& accessKey,
      MNSConnectionToolPtr mnsConnTool)
    : mQueueName(queueName)
    , mEndpoint(endpoint)
    , mAccessId(accessId)
    , mAccessKey(accessKey)
    , mMnsConnTool(mnsConnTool)
{
}

void Queue::getAttributes(QueueAttributes& attributes)
{
    GetQueueAttributesRequest req(mQueueName);
    GetQueueAttributesResponse resp(attributes);
    MNSClient::sendRequest(req, resp, mEndpoint, mAccessId, mAccessKey, mMnsConnTool);
}

void Queue::setAttributes(const QueueAttributes& attributes)
{
    SetQueueAttributesRequest req(mQueueName, attributes);
    SetQueueAttributesResponse resp;
    MNSClient::sendRequest(req, resp, mEndpoint, mAccessId, mAccessKey, mMnsConnTool);
}

void Queue::sendMessage(const std::string& messageBody,
                 SendMessageResponse& resp)
{
    SendMessageRequest req(messageBody, -1, -1);
    req.setQueueName(mQueueName);
    MNSClient::sendRequest(req, resp, mEndpoint, mAccessId, mAccessKey, mMnsConnTool);
}

void Queue::sendMessage(const std::string& messageBody,
                        const int32_t delaySeconds,
                        const int32_t priority,
                        SendMessageResponse& resp)
{
    SendMessageRequest req(messageBody, delaySeconds, priority);
    req.setQueueName(mQueueName);
    MNSClient::sendRequest(req, resp, mEndpoint, mAccessId, mAccessKey, mMnsConnTool);
}

void Queue::batchSendMessage(
    const std::vector<SendMessageItem>& sendMessageItems,
    BatchSendMessageResponse& resp)
{
    BatchSendMessageRequest req(sendMessageItems);
    req.setQueueName(mQueueName);
    MNSClient::sendRequest(req, resp, mEndpoint, mAccessId, mAccessKey, mMnsConnTool);
}

void Queue::peekMessage(Message& message)
{
    PeekMessageRequest req;
    req.setQueueName(mQueueName);
    PeekMessageResponse resp(message);
    MNSClient::sendRequest(req, resp, mEndpoint, mAccessId, mAccessKey, mMnsConnTool);
}

void Queue::batchPeekMessage(const int32_t numOfMessages,
    std::vector<Message>& messages)
{
    BatchPeekMessageRequest req(numOfMessages);
    req.setQueueName(mQueueName);
    BatchPeekMessageResponse resp(messages);
    MNSClient::sendRequest(req, resp, mEndpoint, mAccessId, mAccessKey, mMnsConnTool);
}

void Queue::receiveMessage(Message& message)
{
    ReceiveMessageRequest req;
    req.setQueueName(mQueueName);
    ReceiveMessageResponse resp(message);
    MNSClient::sendRequest(req, resp, mEndpoint, mAccessId, mAccessKey, mMnsConnTool);
}

void Queue::receiveMessage(const int32_t waitSeconds,
                           Message& message)
{
    ReceiveMessageRequest req(waitSeconds);
    req.setQueueName(mQueueName);
    ReceiveMessageResponse resp(message);
    MNSClient::sendRequest(req, resp, mEndpoint, mAccessId, mAccessKey, mMnsConnTool);
}

void Queue::batchReceiveMessage(const int32_t numOfMessages,
                                std::vector<Message>& messages)
{
    BatchReceiveMessageRequest req(numOfMessages);
    req.setQueueName(mQueueName);
    BatchReceiveMessageResponse resp(messages);
    MNSClient::sendRequest(req, resp, mEndpoint, mAccessId, mAccessKey, mMnsConnTool);
}

void Queue::batchReceiveMessage(const int32_t numOfMessages,
                                const int32_t waitSeconds,
                                std::vector<Message>& messages)
{
    BatchReceiveMessageRequest req(numOfMessages, waitSeconds);
    req.setQueueName(mQueueName);
    BatchReceiveMessageResponse resp(messages);
    MNSClient::sendRequest(req, resp, mEndpoint, mAccessId, mAccessKey, mMnsConnTool);
}

void Queue::deleteMessage(const std::string& receiptHandle)
{
    DeleteMessageRequest req(receiptHandle);
    req.setQueueName(mQueueName);
    DeleteMessageResponse resp;
    MNSClient::sendRequest(req, resp, mEndpoint, mAccessId, mAccessKey, mMnsConnTool);
}

void Queue::batchDeleteMessage(const std::vector<std::string>& receiptHandles,
                               BatchDeleteMessageResponse& resp)
{
    BatchDeleteMessageRequest req(receiptHandles);
    req.setQueueName(mQueueName);
    MNSClient::sendRequest(req, resp, mEndpoint, mAccessId, mAccessKey, mMnsConnTool);
}

void Queue::changeMessageVisibility(const std::string& receiptHandle,
                                    const int32_t visibilityTimeout,
                                    ChangeMessageVisibilityResponse& resp)
{
    ChangeMessageVisibilityRequest req(receiptHandle, visibilityTimeout);
    req.setQueueName(mQueueName);
    MNSClient::sendRequest(req, resp, mEndpoint, mAccessId, mAccessKey, mMnsConnTool);
}
