package mns;

import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

import com.alibaba.fastjson.JSON;

import com.aliyun.mns.client.CloudAccount;
import com.aliyun.mns.client.CloudQueue;
import com.aliyun.mns.model.Message;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import event.BaseEvent;
import event.EventSubscriber;

/**
 * <p>
 *     该类是对MNS进行封装的接收网卡事件的客户端类，该类定义了一个线程负责接收消息，一个线程池负责处理消息。
 * 在实际生产中，该对象为单例，只被初始化一次，可用于接收ecs事件发送的各种消息。根据不同云监控事件名称分别定义并实现EventSubscriber。
 * </p>
 * <p>
 * 说明：
 *  采用MNS最佳实践的长轮询方式接收消息。https://help.aliyun.com/document_detail/34478.html
 *      <li>queueName 为MNS的队列名，建议网卡操作完成事件单独一个队列</li>
 *      <li>account 为初始化MNS Client云账户对象</li>
 *      <li>receiveEventThread 线程专门负责拉取消息</li>
 *      <li>subscriberThreads 线程池专门负责处理消息</li>
 *      <li>running 用来标识是否正常处理消息</li>
 *      <li>exit 用来标识消息是否已经全部提交线程池后正常退出</li>
 * </p>
 * <p>
 * 限制：
 *  当前代码为示例代码，如需用于生产，请根据自己业务情况进行修改。
 *      <li>请根据实际业务规模、处理时长等调整线程池参数、对线程进行监控</li>
 *      <li>示例为单线程拉取，可以根据实际业务规模进行修改</li>
 *      <li>示例未考虑消息处理失败的场景，处理任务需保证消息处理的正确性</li>
 *      <li>日志处理等请根据应用使用框架进行修改</li>
 *      <li>需根据云监控事件名称进行定义并注册一个EventSubscriber</li>
 * </p>
 */
public class EventClient {
    private static final int WAIT_SECONDS = 30;
    private static EventClient instance;
    private ExecutorService subscriberThreads;
    private Thread receiveEventThread;
    private Map<String/*event name*/, EventSubscriber> subscriberMap;
    private CloudQueue cloudQueue;
    private String queueName;
    private volatile boolean running;
    private volatile boolean exit;

    /**
     * 初始化
     * @param account
     * @param queueName
     */
    private EventClient(CloudAccount account, String queueName) {
        if (account == null) {
            throw new IllegalArgumentException("account is null");
        }
        if (queueName == null) {
            throw new IllegalArgumentException("queueName is null");
        }

        this.subscriberMap = new HashMap<>();
        this.subscriberThreads = new ThreadPoolExecutor(1, 5,
            0, TimeUnit.MILLISECONDS, new LinkedBlockingQueue<Runnable>(100),
            new ThreadFactoryBuilder().setNameFormat("vmFlow_threadPoolName").build(),
            new ThreadPoolExecutor.AbortPolicy());
        this.receiveEventThread = new Thread(new ReceiveMessageRunner());
        com.aliyun.mns.client.MNSClient mnsClient = account.getMNSClient();
        this.cloudQueue = mnsClient.getQueueRef(queueName);
        this.queueName = queueName;
    }

    /**
     * 单例，初始化EventClient
     * @param account
     * @param queueName
     * @return
     */
    public static synchronized EventClient instance(CloudAccount account, String queueName) {
        if (instance == null) {
            instance = new EventClient(account, queueName);
        } else {
            if (!instance.queueName.equals(queueName)) {
                throw new IllegalArgumentException("ambiguous queueName");
            }
        }
        return instance;
    }

    /**
     * 注册不同类型的事件
     * @param subscriber
     * @return
     */
    public EventClient register(EventSubscriber subscriber) {
        if (subscriber == null) {
            throw new IllegalArgumentException("subscriber is null");
        }
        this.subscriberMap.putIfAbsent(subscriber.subscribedToEventType(), subscriber);
        return this;
    }

    public synchronized void start() {
        if (running) {
            return;
        }
        running = true;
        receiveEventThread.start();
    }

    public synchronized void stop() {
        if (!running) {
            return;
        }
        running = false;
        while (true) {
            if (exit) {
                receiveEventThread.interrupt();
                subscriberThreads.shutdown();
                break;
            }
        }
    }

    /**
     * 通过长轮训拉取消息
     * @return
     */
    private Message receiveMessage() {
        try {
            System.out.println("Thread KEEP Polling!");
            Message message = cloudQueue.popMessage(WAIT_SECONDS);
            return message;
        } catch (Exception e) {
            // it could be network exception
            System.out.println("Exception Happened when popMessage: " + e);
        }
        return null;
    }

    /**
     * 接收消息，并提交线程池进行业务处理
     */
    private class ReceiveMessageRunner implements Runnable {
        @Override
        public void run() {
            while (true) {
                if (!running) {
                    break;
                }
                Message message = receiveMessage();
                if (message == null) {
                    continue;
                }

                try {
                    BaseEvent baseEvent = JSON.parseObject(
                        message.getMessageBodyAsBase64(), BaseEvent.class);
                    EventSubscriber subscriber = subscriberMap.get(baseEvent.getName());
                    if (subscriber == null) {
                        System.out.println(String.format("Missing subscriber %s! You need fix it.", baseEvent.getName()));
                        continue;
                    }
                    subscriberThreads.execute(() -> subscriber.handleEvent(baseEvent));
                    cloudQueue.deleteMessage(message.getReceiptHandle());
                } catch (Exception e) {
                    // it could be network exception
                    System.out.println("Exception Happened when popMessage: " + e);
                }
            }
            exit = true;
        }
    }
}
