/*
 * Decompiled with CFR 0.152.
 */
package com.alicloud.openservices.tablestore.tunnel.worker;

import com.alicloud.openservices.tablestore.ClientException;
import com.alicloud.openservices.tablestore.TableStoreException;
import com.alicloud.openservices.tablestore.TunnelClientInterface;
import com.alicloud.openservices.tablestore.core.utils.Preconditions;
import com.alicloud.openservices.tablestore.model.tunnel.internal.Channel;
import com.alicloud.openservices.tablestore.model.tunnel.internal.ConnectTunnelRequest;
import com.alicloud.openservices.tablestore.model.tunnel.internal.ConnectTunnelResponse;
import com.alicloud.openservices.tablestore.model.tunnel.internal.HeartbeatRequest;
import com.alicloud.openservices.tablestore.model.tunnel.internal.HeartbeatResponse;
import com.alicloud.openservices.tablestore.model.tunnel.internal.ShutdownTunnelRequest;
import com.alicloud.openservices.tablestore.tunnel.worker.ChannelDialer;
import com.alicloud.openservices.tablestore.tunnel.worker.ChannelProcessFactory;
import com.alicloud.openservices.tablestore.tunnel.worker.IChannelDialer;
import com.alicloud.openservices.tablestore.tunnel.worker.IChannelProcessorFactory;
import com.alicloud.openservices.tablestore.tunnel.worker.ITunnelWorker;
import com.alicloud.openservices.tablestore.tunnel.worker.TunnelClientConfig;
import com.alicloud.openservices.tablestore.tunnel.worker.TunnelStateMachine;
import com.alicloud.openservices.tablestore.tunnel.worker.TunnelWorkerConfig;
import com.alicloud.openservices.tablestore.tunnel.worker.TunnelWorkerStatus;
import java.util.Date;
import java.util.List;
import java.util.Random;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.Semaphore;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicReference;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TunnelWorker
implements ITunnelWorker {
    private static final Logger LOG = LoggerFactory.getLogger(TunnelWorker.class);
    private static final int CORE_POOL_SIZE = 2;
    private static final int WORKER_RANDOM_RETRY_MILLIS = 10000;
    private String tunnelId;
    private String clientId;
    private TunnelWorkerConfig workerConfig;
    private TunnelClientInterface client;
    private TunnelStateMachine stateMachine;
    private IChannelDialer channelDialer;
    private AtomicReference<TunnelWorkerStatus> workerStatus = new AtomicReference();
    private Date lastHeartbeatTime;
    private ScheduledExecutorService heartbeatExecutor;
    private IChannelProcessorFactory factory;

    public TunnelWorker(String tunnelId, TunnelClientInterface client, TunnelWorkerConfig workerConfig) {
        this(tunnelId, client, workerConfig, null);
    }

    public TunnelWorker(String tunnelId, TunnelClientInterface client, TunnelWorkerConfig workerConfig, IChannelProcessorFactory factory) {
        Preconditions.checkArgument(tunnelId != null && !tunnelId.isEmpty(), "The tunnel id should not be null or empty.");
        Preconditions.checkNotNull(client, "Tunnel client cannot be null.");
        Preconditions.checkNotNull(workerConfig, "Tunnel worker workerConfig cannot be null.");
        this.factory = factory;
        if (this.factory == null) {
            Preconditions.checkNotNull(workerConfig.getChannelProcessor(), "Channel Processor cannot be null.");
        }
        this.init(tunnelId, client, workerConfig);
    }

    private void init(String tunnelId, TunnelClientInterface client, TunnelWorkerConfig config) {
        LOG.info("Initial tunnel worker, tunnelId: {}", (Object)tunnelId);
        this.tunnelId = tunnelId;
        this.client = client;
        if (config.getMaxChannelParallel() > 0) {
            config.setMaxChannelSemaphore(new Semaphore(config.getMaxChannelParallel(), true));
        }
        this.workerConfig = config;
        this.workerStatus.set(TunnelWorkerStatus.WORKER_READY);
        this.lastHeartbeatTime = new Date();
        this.heartbeatExecutor = Executors.newScheduledThreadPool(2, new ThreadFactory(){
            private final AtomicInteger counter = new AtomicInteger(0);

            @Override
            public Thread newThread(Runnable r) {
                return new Thread(r, "tunnel-heartbeat-scheduled-" + this.counter.getAndIncrement());
            }
        });
    }

    private void connect() {
        if (!this.workerStatus.compareAndSet(TunnelWorkerStatus.WORKER_READY, TunnelWorkerStatus.WORKER_STARTED)) {
            throw new ClientException(String.format("Tunnel worker has already been %s status", this.workerStatus));
        }
        while (true) {
            try {
                TunnelClientConfig conf = new TunnelClientConfig(this.workerConfig.getHeartbeatTimeoutInSec(), this.workerConfig.getClientTag());
                ConnectTunnelRequest request = new ConnectTunnelRequest(this.tunnelId, conf);
                ConnectTunnelResponse resp = this.client.connectTunnel(request);
                this.clientId = resp.getClientId();
                this.channelDialer = new ChannelDialer(this.client, this.workerConfig);
                if (this.factory == null) {
                    this.factory = new ChannelProcessFactory(this.workerConfig);
                }
                this.stateMachine = new TunnelStateMachine(this.tunnelId, this.clientId, this.channelDialer, this.factory, this.client);
                this.stateMachine.setEnableClosingChannelDetect(this.workerConfig.isEnableClosingChannelDetect());
                LOG.info("Connect tunnel success, RequestId: {}, clientId: {}, tunnelId: {}", new Object[]{resp.getRequestId(), this.clientId, this.tunnelId});
                break;
            }
            catch (TableStoreException te) {
                LOG.warn("Connect tunnel failed, tunnel id {}, error detail {}", (Object)this.tunnelId, (Object)te.toString());
                if (this.isTunnelInvalid(te.getErrorCode())) {
                    LOG.error("Tunnel is expired or invalid, tunnel worker will be halted.");
                    this.workerStatus.set(TunnelWorkerStatus.WORKER_HALT);
                    throw te;
                }
            }
            catch (Exception e) {
                LOG.warn("Connect tunnel failed, tunnel id {}, error detail {}", (Object)this.tunnelId, (Object)e.toString());
            }
            try {
                Thread.sleep(new Random(System.currentTimeMillis()).nextInt(10000) + 1);
            }
            catch (Exception e) {
                LOG.warn("Reconnect worker error, error detail: {}", (Object)e.toString());
            }
        }
    }

    private boolean isTunnelInvalid(String errorCode) {
        return errorCode.equals("OTSTunnelExpired") || errorCode.equals("OTSParameterInvalid");
    }

    private String channelsToString(List<Channel> channels) {
        StringBuilder sb = new StringBuilder();
        sb.append("[");
        for (Channel channel : channels) {
            sb.append(channel);
        }
        sb.append("]");
        return sb.toString();
    }

    @Override
    public void connectAndWorking() throws Exception {
        this.connect();
        this.heartbeatExecutor.scheduleAtFixedRate(new Heartbeat(), 0L, this.workerConfig.getHeartbeatIntervalInSec(), TimeUnit.SECONDS);
        Runtime.getRuntime().addShutdownHook(new Thread(new Runnable(){

            @Override
            public void run() {
                LOG.warn("Unexpected shutdown, do resources clear.");
                TunnelWorker.this.shutdown();
            }
        }));
    }

    @Override
    public void shutdown() {
        this.shutdown(true);
    }

    private void shutdown(boolean isHalt) {
        if (this.workerStatus.get().equals((Object)TunnelWorkerStatus.WORKER_ENDED) || this.workerStatus.get().equals((Object)TunnelWorkerStatus.WORKER_HALT)) {
            LOG.info("Tunnel worker has already been {} status, skip shutdown logic.", this.workerStatus);
            return;
        }
        if (this.channelDialer != null) {
            LOG.info("Shutdown channel dialer");
            this.channelDialer.shutdown();
        }
        if (this.stateMachine != null) {
            LOG.info("Shutdown tunnel state machine.");
            this.stateMachine.close();
        }
        if (isHalt && this.heartbeatExecutor != null) {
            LOG.info("Shutdown heartbeat executor.");
            this.heartbeatExecutor.shutdownNow();
            try {
                if (this.heartbeatExecutor.awaitTermination(5L, TimeUnit.SECONDS)) {
                    LOG.info("Heartbeat executor termination success.");
                } else {
                    LOG.warn("Heartbeat executor termination until timeout");
                }
            }
            catch (InterruptedException e) {
                LOG.warn("Wait heartbeat executor termination failed", (Throwable)e);
            }
        }
        try {
            LOG.info("Shutdown tunnel, tunnelId: {}, clientId: {}", (Object)this.tunnelId, (Object)this.clientId);
            this.client.shutdownTunnel(new ShutdownTunnelRequest(this.tunnelId, this.clientId));
        }
        catch (Exception e) {
            LOG.warn("Shutdown tunnel failed, tunnelId: {}, clientId: {}", new Object[]{this.tunnelId, this.clientId, e});
        }
        LOG.info("Tunnel worker is ended.");
        if (isHalt) {
            this.workerStatus.set(TunnelWorkerStatus.WORKER_HALT);
        } else {
            this.workerStatus.set(TunnelWorkerStatus.WORKER_ENDED);
        }
    }

    private class Heartbeat
    implements Runnable {
        private Heartbeat() {
        }

        @Override
        public void run() {
            if (((TunnelWorkerStatus)((Object)TunnelWorker.this.workerStatus.get())).equals((Object)TunnelWorkerStatus.WORKER_ENDED)) {
                TunnelWorker.this.workerStatus.set(TunnelWorkerStatus.WORKER_READY);
                TunnelWorker.this.lastHeartbeatTime = new Date();
                TunnelWorker.this.connect();
            }
            try {
                Date now = new Date();
                if (TunnelWorker.this.lastHeartbeatTime.compareTo(now) != 0 && now.getTime() - TunnelWorker.this.lastHeartbeatTime.getTime() > TimeUnit.SECONDS.toMillis(TunnelWorker.this.workerConfig.getHeartbeatTimeoutInSec())) {
                    LOG.error("Tunnel client heartbeat timeout, lastHeartbeatTime: {}.", (Object)TunnelWorker.this.lastHeartbeatTime);
                    TunnelWorker.this.shutdown(false);
                    return;
                }
                LOG.info("Begin batch get channels.");
                List<Channel> currChannels = TunnelWorker.this.stateMachine.batchGetChannels();
                HeartbeatRequest request = new HeartbeatRequest(TunnelWorker.this.tunnelId, TunnelWorker.this.clientId, currChannels);
                HeartbeatResponse resp = TunnelWorker.this.client.heartbeat(request);
                TunnelWorker.this.lastHeartbeatTime = new Date();
                List<Channel> targetChannels = resp.getChannels();
                LOG.info("Begin batch update channels, RequestId: {}, num: {}, detail: {}.", new Object[]{resp.getRequestId(), targetChannels.size(), TunnelWorker.this.channelsToString(targetChannels)});
                TunnelWorker.this.stateMachine.batchUpdateChannels(targetChannels);
            }
            catch (TableStoreException te) {
                LOG.warn("Heartbeat error, TableStore Exception: {}.", (Object)te.toString());
                if (TunnelWorker.this.isTunnelInvalid(te.getErrorCode())) {
                    LOG.error("Tunnel is expired or invalid, tunnel worker will be halted.");
                    TunnelWorker.this.shutdown(true);
                }
            }
            catch (Throwable e) {
                LOG.warn("Heartbeat error, Throwable: {}", (Object)e.toString());
            }
        }
    }
}

