/*
 * Copyright 2024 Alibaba Cloud, Inc. or its affiliates.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package sam

/*
#cgo CFLAGS:  -I./include
#cgo LDFLAGS: -L./libs/x86_64 -lsam -Wl,-rpath,lib
#include <stdlib.h>
#include "sam_api_ext.h"
*/
import "C"

import (
	"fmt"
	"os"
	"log"
	"syscall"
	"unsafe"

	"demo/pkg/meta"             //FIXME
	"demo/pkg/crypto"           //FIXME
	"golang.org/x/sys/unix"
)

const (
	productName string = "Demo_Product_Name"
	deviceName  string = "Demo_Device_Name"
)

type SamCryptoEngine struct {
        // Sam Content
	contextPtr *C.sam_context

	// File Password used to derive content key
	passwd []byte

	// Secure Storage Path used to save license
	StoragePath string

	// Device Unique ID
	DeviceUid string

	BlockSize uint32
}

func (s *SamCryptoEngine) EncryptName(dirEntry *crypto.CipherEntry, cname string) (string, syscall.Errno) {
	return cname, 0
}

func (s *SamCryptoEngine) DecryptName(dirEntry *crypto.CipherEntry, cname string) (string, bool, syscall.Errno) {
	return cname, false, 0
}

func (s *SamCryptoEngine) DecryptSymlinkTarget(ctarget string) (string, error) {
	return ctarget, nil
}

func (s *SamCryptoEngine) CipherSizeToPlainSize(cPath string, size uint64) (uint64, error) {
	var result C.sam_result
	var dataOff C.uint64_t
	var plainSize uint64
	var mmapSize int

	if s.contextPtr == nil {
		return 0, fmt.Errorf("Sam Context error")
	}

	file, err := os.Open(cPath)
	if err != nil {
		return 0, err
	}
	defer file.Close()

	fileInfo, err := file.Stat()
	if err != nil {
		return 0, err
	}

	/* set mmap size to 4K */
	mmapSize = 0x1000
	if fileInfo.Size() < int64(mmapSize) {
		mmapSize = int(fileInfo.Size())
	}

	mmaped, err := unix.Mmap(int(file.Fd()), 0, mmapSize, syscall.PROT_READ, syscall.MAP_SHARED)
	if err != nil {
		return 0, err
	}
	defer unix.Munmap(mmaped)

	contData := (*C.uint8_t)(unsafe.Pointer(&mmaped[0]))
	contSize := C.uint64_t(mmapSize)
	dataOffPtr := (*C.uint64_t)(unsafe.Pointer(&dataOff))
	result = C.sam_get_content_data_offset(s.contextPtr, contData, contSize, dataOffPtr)
	if result != C.SAM_SUCCESS {
		return 0, fmt.Errorf("C.sam_get_content_data_offset error: 0x%x", result)
	}

	if size <= uint64(dataOff) {
		return 0, fmt.Errorf("CipherSize is corrupted, size %d dataOff %d", size, dataOff)
	}
	plainSize = size - uint64(dataOff)

	return plainSize, nil
}

func (s *SamCryptoEngine) ReadAtCipherText(inode meta.Ino, file *os.File, buf []byte, off int64) (int, error) {
	var result C.sam_result
	var size int = len(buf)

	if s.contextPtr == nil {
		return 0, fmt.Errorf("Sam Context error")
	}

	passwordPtr := (* C.char)(C.CString(string(s.passwd)))
	outData := (*C.uint8_t)(unsafe.Pointer(&buf[0]))
	outSize := C.uint64_t(size)
	result = C.sam_on_block_decryption(s.contextPtr, passwordPtr,
		C.int(file.Fd()), C.uint64_t(off), C.uint64_t(size), outData, outSize)
	if result != C.SAM_SUCCESS {
		return 0, fmt.Errorf("C.sam_on_block_decryption error: 0x%x", result)
	}
	defer C.free(unsafe.Pointer(passwordPtr))

	return size, nil
}

func (s *SamCryptoEngine) Wipe() {
	if (s.contextPtr != nil) {
		C.sam_final_context(s.contextPtr)
		s.contextPtr = nil
	}

	if s.passwd != nil {
		for i := 0; i < len(s.passwd); i = i + 1 {
			s.passwd[i] = 0
		}
		s.passwd = nil
	}
}

func NewSamCryptoEngine(passwd []byte, blockSize uint32, samDir, samDevUid string) (crypto.CryptoEngine, func()) {
	var result C.sam_result
	var config C.sam_config
	var context C.sam_context

	if len(passwd) == 0 {
		log.Fatalf("Sam crypto mode requires non-empty password")
        }

        engine := &SamCryptoEngine{
		StoragePath:  samDir,
		DeviceUid:    samDevUid,
		BlockSize:    blockSize,
        }

	config.sst_path = (*C.char)(unsafe.Pointer(C.CString(engine.StoragePath)))
	config.dev_uuid = (*C.char)(unsafe.Pointer(C.CString(engine.DeviceUid)))
	config.timeout_ms = 1000
	configPtr := (*C.sam_config)(unsafe.Pointer(&config))
	result = C.sam_set_config(configPtr)
	if result != C.SAM_SUCCESS {
		log.Fatalf("C.sam_set_config error: 0x%x", result)
	}

	defer C.free(unsafe.Pointer(config.sst_path))
	defer C.free(unsafe.Pointer(config.dev_uuid))

	productNamePtr := (*C.char)(unsafe.Pointer(C.CString(productName)))
	deviceNamePtr := (*C.char)(unsafe.Pointer(C.CString(deviceName)))
	contextPtr := (*C.sam_context)(unsafe.Pointer(&context))
	result = C.sam_init_context(productNamePtr, deviceNamePtr, contextPtr)
	if result != C.SAM_SUCCESS {
		log.Fatalf("C.sam_init_context error: 0x%x", result)
	}

	defer C.free(unsafe.Pointer(productNamePtr))
	defer C.free(unsafe.Pointer(deviceNamePtr))

	engine.contextPtr = contextPtr

        engine.passwd = make([]byte, len(passwd))
        copy(engine.passwd, passwd)

	return engine, func() { engine.Wipe() }
}
