/*
 * Copyright (C) 2018-2020 Alibaba Group Holding Limited
 */

#ifndef __SAM_API_H__
#define __SAM_API_H__

#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

#define IOT_SAM_MAX_EXTRAS_SIZE     256
#define IOT_SAM_MAX_LIC_ID_SIZE     48

/* API error code definitions */
#define SAM_SUCCESS                 (0x00000000)  /* The operation was successful */
#define SAM_ERROR_GENERIC           (0xFFFF0001)  /* The generic error */
#define SAM_ERROR_BAD_PARAMETERS    (0xFFFF0002)  /* Input parameters are invalid */
#define SAM_ERROR_SHORT_BUFFER      (0xFFFF0003)  /* The supplied buffer is too short for output */
#define SAM_ERROR_EXCESS_DATA       (0xFFFF0004)  /* Too much data for the requested operation */
#define SAM_ERROR_OUT_OF_MEMORY     (0xFFFF0005)  /* Out of memory */
#define SAM_ERROR_COMMUNICATION     (0xFFFF0006)  /* Communication error */
#define SAM_ERROR_NOT_SUPPORTED     (0xFFFF0007)  /* The request operation is not supported */
#define SAM_ERROR_ITEM_NOT_FOUND    (0xFFFF0008)  /* The item is not exist */
#define SAM_ERROR_NOT_IMPLEMENTED   (0xFFFF0009)  /* The request operation is not implemented */
#define SAM_ERROR_TIMEOUT           (0xFFFF000A)  /* Communication timeout */
#define SAM_ERROR_EXPIRED           (0xFFFF000B)  /* The license has expired */
#define SAM_ERROR_CANCELED          (0xFFFF000C)  /* Operation has been canceled */
#define SAM_ERROR_LICENSE_ERROR     (0xFFFF000D)  /* The license error is returned */
#define SAM_ERROR_LICENSE_REVOKED   (0xFFFF000F)  /* The license has been revoked */

/* SAM item type definitions */
#define SAM_ITEM_TYPE_DICT          (0x01)  /* The dictionary */

#define SAM_MAX_AUTH_CODE_LEN        256

#define SAM_MAX_LIC_OTP_LEN          512

typedef uint32_t sam_result;
typedef void *sam_context_handle;
typedef void *sam_session_handle;

typedef struct {
    sam_context_handle imp;
} sam_context;

typedef struct {
    sam_session_handle imp;
} sam_session;

typedef struct {
    char *sst_path;
    char *dev_uuid;
    uint32_t timeout_ms;
} sam_config;

typedef int (*pub_handle_t)(const char *topic,
    const uint8_t *msg, uint32_t msg_size, void *channel, const void *data);

/**
 * @brief Set global configs.
 *
 * @param [in] config: Specify the configs:
 *             - sst_path: the proposed secure storage path, zero-terminated, less than 128 bytes.
 *             - dev_uuid: the proposed device unique ID, zero-terminated, less than 100 bytes.
 *             - timeout_ms: request timeout miliseconds, [1000, 20000].
 *
 * @see error code definitions.
 */
sam_result sam_set_config(sam_config *config);

/**
 * @brief Get library version.
 *
 * @param [out] version: A array to store version string.
 *
 * @see None.
 */
void sam_get_version(char version[16]);

/**
 * @brief Initializes a new SAM context.
 *
 * @param [in] product_name: Specify product name, zero-terminated, less than 48 bytes.
 * @param [in] device_name: Specify device unique name, zero-terminated, less than 48 bytes.
 * @param [out] context: The context structure which is to be initialized.
 *
 * @see error code definitions.
 */
sam_result sam_init_context(char *product_name, char *device_name, sam_context *context);

/**
 * @brief Finalizes an initialized SAM context.
 *
 * @param [in] context: An initialized context structure which is to be finalized.
 *
 * @see None.
 */
void sam_final_context(sam_context *context);

/**
 * @brief Set product secret - optional.
 *
 * @param [in] context: Pointer to an initialized SAM context.
 * @param [in] product_secret: Specify product secret, zero-terminated, less than 48 bytes.
 *
 * @see error code definitions.
 */
sam_result sam_set_product_secret(sam_context *context, char *product_secret);

/**
 * @brief Register communication publish handler.
 *
 * @param [in] context: Pointer to an initialized SAM context.
 * @param [in] publish_handle: Pointer to a publish handler.
 * @param [in] channel: Communication channel, which is used by pushlish handler.
 *
 * @see error code definitions.
 */
sam_result sam_set_pub_handle(sam_context *context, pub_handle_t pub_handle, void *channel);

/**
 * @brief Set communication publish topic.
 *
 * @param [in] context: Pointer to an initialized SAM context.
 * @param [in] topic_name: Specify publish topic, zero-terminated, less than 128 bytes.
 *
 * @see error code definitions.
 */
sam_result sam_set_pub_topic(sam_context *context, char *topic_name);

/**
 * @brief Process received license message.
 *
 * @param [in] context: Pointer to an initialized SAM context.
 * @param [in] msg: License message hexadecimal data.
 * @param [in] msg_size: License message size, less than 1024 bytes.
 *
 * @see error code definitions.
 */
sam_result sam_on_message(sam_context *context, uint8_t *msg, uint32_t msg_size);

/**
 * @brief Get content data offset.
 *
 * @param [in] context: Pointer to an initialized SAM context.
 * @param [in] data: The content buffer.
 * @param [in] data_len: The length of content buffer.
 * @param [out] data_off: The data item offset to the content buffer.
 *
 * @see error code definitions.
 */
sam_result sam_get_content_data_offset(sam_context *context,
                   uint8_t *data, uint64_t data_len, uint64_t *data_off);

/**
 * @brief Get license id in content.
 *
 * @param [in] context: Pointer to an initialized SAM context.
 * @param [in] data: The content buffer.
 * @param [in] data_len: The length of content buffer.
 * @param [out] lic_id: The buffer to save license id.
 * @param [in] size: The buffer size, no less than IOT_SAM_MAX_LIC_ID_SIZE bytes.
 *
 * @see error code definitions.
 */
sam_result sam_get_content_license_id(sam_context *context,
                   uint8_t *data, uint64_t data_len, uint8_t *lic_id, uint32_t size);

/**
 * @brief Open a new session.
 *
 * @param [in] context: Pointer to an initialized SAM context.
 * @param [out] session: Pointer to a session structure to open.
 * @param [in] license_name: Specify license ID, zero-terminated, less than 48 bytes.
 *
 * @see error code definitions.
 */
sam_result sam_open_session(sam_context *context, sam_session *session, char *license_name);

/**
 * @brief Close a session which has been opened.
 *
 * @param [in] session: The opened session to close.
 *
 * @see None.
 */
void sam_close_session(sam_session *session);

/**
 * @brief Set publish necessary data - Optional.
 *
 * @param [in] pub_data: Specify necessary data required to publish.
 *
 * @see error code definitions.
 */
sam_result sam_set_pub_data(sam_session *session, void *pub_data);

/**
 * @brief Execute a cancel request in the specified session.
 */
void sam_cancel_request(sam_session *session);

/**
 * @brief Get the one-time provisioning authentication code.
 *
 * @param [in] session: The open seesion in which license will be operated.
 * @param [in] token: The provisioning token, terminated with '\0'.
 * @param [out] auth_code: the output buffer, containing auth code.
 * @param [inout] auth_code_len: in - the length of buffer, no less than SAM_MAX_AUTH_CODE_LEN.
 *                               out - the actual length of auth code.
 *
 * @see error code definitions.
 */
sam_result sam_get_otp_auth_code(sam_session *session,
                   const char *token, uint8_t *auth_code, uint32_t *auth_code_len);

/**
 * @brief Verify auth code and generate license otp data.
 *
 * @param [in] session: The open seesion in which license will be operated.
 * @param [in] auth_code: auth code of client.
 * @param [in] auth_code_len: the length of auth code.
 * @param [out] otp_data: the output buffer, containing otp data.
 * @param [inout] otp_len: in - the length of buffer, no less than SAM_MAX_LIC_OTP_LEN.
 *                         out - the actual length of otp data.
 *
 * @see error code definitions.
 */
sam_result sam_verify_and_get_otp_data(sam_session *session,
               uint8_t *auth_code, uint32_t auth_code_len, uint8_t *otp_data, uint32_t *otp_len);

/**
 * @brief Load the otp data into SAM secure storage.
 *
 * @param [in] session: The open seesion in which license will be operated.
 * @param [in] otp_data: the license otp data, which is got from service.
 * @param [in] otp_len: in - the length of license otp data.
 *
 * @see error code definitions.
 */
sam_result sam_load_lic_otp_data(sam_session *session, uint8_t *otp_data, uint32_t otp_len);

/**
 * @brief Get license provisiong status.
 *
 * @param [in] session: The open seesion in which license will be operated.
 * @param [out] is_prov: The license provisioning status:
 *                       0 - not been provisioned.
 *                       1 - has been provisioned.
 *
 * @see error code definitions.
 */
sam_result sam_get_lic_prov_stat(sam_session *session, uint32_t *is_prov);

/**
 * @brief Set license into SAM secure storage.
 *
 * @param [in] session: The open seesion in which license will be operated.
 * @param [in] info: The license info to be saved.
 * @param [in] size: The size of license info.
 *
 * @see error code definitions.
 */
sam_result sam_set_lic_info(sam_session *session, uint8_t *info, uint32_t size);

/**
 * @brief Get extras info in license.
 *
 * @param [in] session: The open seesion in which license will be operated.
 * @param [out] extras: The buffer to save extras info.
 * @param [in] size: The buffer size, no less than IOT_SAM_MAX_EXTRAS_SIZE bytes.
 *
 * @see error code definitions.
 */
sam_result sam_get_lic_extras(sam_session *session, uint8_t *extras, uint32_t size);

/**
 * @brief Get license error code when SAM_ERROR_LICENSE_ERROR is returned.
 *
 * @param [in] session: The open seesion in which license will be operated.
 * @param [out] lic_err: license error code:
 *              205 : input parameters are invalid.
 *              206 : product key is invalid.
 *              207 : license id is invalid.
 *              208 : signature is invalid.
 *              209 : license has been revoked.
 *              210 : license time has expired.
 *              211 : license count has exipred.
 *              212 : license no quota.
 *              299 : generic error.
 *
 * @see error code definitions.
 */
sam_result sam_get_lic_error_code(sam_session *session, uint32_t *lic_err);

/**
 * @brief check license rights.
 *
 * @param [in] session: The open seesion in which license will be operated.
 *
 * @see error code definitions.
 */
sam_result sam_chk_lic_rights(sam_session *session);

/**
 * @brief File-layered cipher data decryption.
 *
 * @param [in] session: The open seesion in which license has been binded.
 * @param [in] in: The input cipher data.
 * @param [in] in_len: The length of input cipher data.
 * @param [out] out: The buffer to contain plaintext data.
 * @param [inout] out_len: in - The buffer length;
                           out - The actual plaintext length.
 *
 * @see error code definitions.
 */
sam_result sam_on_decryption(sam_session *session,
                  uint8_t *in, uint64_t in_len, uint8_t *out, uint64_t *out_len);

/**
 * @brief execute item cipher data decryption.
 *
 * @param [in] session: The open seesion in which license has been binded.
 * @param [in] type: The item type.
 * @param [in] head: The item head info.
 * @param [in] head_len: The length of item head info.
 * @param [in] in: The input item cipher data.
 * @param [in] in_len: The length of item cipher data.
 * @param [out] out: The buffer to contain plaintext data.
 * @param [inout] out_len: in - The buffer length;
                           out - The actual plaintext length.
 *
 * @see error code definitions.
 */
sam_result sam_on_item_decryption(sam_session *session,
                  uint32_t type, uint8_t *head_info, uint32_t head_len,
                  uint8_t *in, uint32_t in_len, uint8_t *out, uint32_t *out_len);

#ifdef __cplusplus
}
#endif

#endif /* __SAM_API_H__ */
