package com.alibaba.ververica.cep.demo;
import com.alibaba.ververica.cep.demo.event.EventBoundedOutOfOrdernessWatermarks;
import com.alibaba.ververica.cep.demo.event.Match_results;
import org.apache.flink.api.common.eventtime.WatermarkStrategy;
import org.apache.flink.api.common.typeinfo.TypeHint;
import org.apache.flink.api.common.typeinfo.TypeInformation;
import org.apache.flink.api.java.functions.KeySelector;
import org.apache.flink.api.java.tuple.Tuple2;
import org.apache.flink.api.java.utils.MultipleParameterTool;
import org.apache.flink.cep.CEP;
import org.apache.flink.cep.TimeBehaviour;
import org.apache.flink.connector.jdbc.JdbcConnectionOptions;
import org.apache.flink.connector.jdbc.JdbcExecutionOptions;
import org.apache.flink.connector.jdbc.JdbcSink;
import org.apache.flink.connector.jdbc.JdbcStatementBuilder;
import org.apache.flink.connector.kafka.source.KafkaSource;
import org.apache.flink.connector.kafka.source.enumerator.initializer.OffsetsInitializer;
import org.apache.flink.streaming.api.datastream.*;
import org.apache.flink.streaming.api.environment.StreamExecutionEnvironment;
import org.apache.flink.streaming.api.functions.ProcessFunction;
import org.apache.flink.util.Collector;
import com.alibaba.ververica.cep.demo.dynamic.JDBCPeriodicPatternProcessorDiscovererFactory;
import com.alibaba.ververica.cep.demo.event.Event;
import com.alibaba.ververica.cep.demo.event.EventDeSerializationSchema;
import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.time.Duration;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import static com.alibaba.ververica.cep.demo.Constants.*;

public class CepDemo {

    public static void checkArg(String argName, MultipleParameterTool params) {
        if (!params.has(argName)) {
            throw new IllegalArgumentException(argName + " must be set!");
        }
    }

    // Parsing the output string returns an EVent object
    private static Match_results parseOutput(String output) {
        String rule = "\\(id, version\\): \\((\\d+), (\\d+)\\).*?Event\\((\\d+), (\\w+), (\\d+), (\\d+)";
        Pattern pattern = Pattern.compile(rule);
        Matcher matcher = pattern.matcher(output);
        if (matcher.find()) {
            return new Match_results(Integer.parseInt(matcher.group(1)), Integer.parseInt(matcher.group(2)), Integer.parseInt(matcher.group(3)), matcher.group(4), Integer.parseInt(matcher.group(6)));
        }
        return null;
    }

    public static void main(String[] args) throws Exception {
        // Process args
        final MultipleParameterTool params = MultipleParameterTool.fromArgs(args);

        checkArg(KAFKA_BROKERS_ARG, params);
        checkArg(INPUT_TOPIC_ARG, params);
        checkArg(INPUT_TOPIC_GROUP_ARG, params);
        checkArg(JDBC_URL_ARG, params);
        checkArg(TABLE_NAME_ARG, params);
        checkArg(JDBC_INTERVAL_MILLIS_ARG, params);

        checkArg(USING_EVENT_TIME, params);

        // Set up the streaming execution environment
        final StreamExecutionEnvironment env = StreamExecutionEnvironment.getExecutionEnvironment();

        // Build Kafka source with new Source API based on FLIP-27
        KafkaSource<Event> kafkaSource =
                KafkaSource.<Event>builder()
                        .setBootstrapServers(params.get(KAFKA_BROKERS_ARG))
                        .setTopics(params.get(INPUT_TOPIC_ARG))
                        .setStartingOffsets(OffsetsInitializer.latest())
                        .setGroupId(params.get(INPUT_TOPIC_GROUP_ARG))
                        .setDeserializer(new EventDeSerializationSchema())
                        .build();
        // DataStream Source
        DataStreamSource<Event> source =
                env.fromSource(
                        kafkaSource,
                        WatermarkStrategy.<Event>forBoundedOutOfOrderness(Duration.ofSeconds(5))
                                .withTimestampAssigner((event, ts) -> event.getEventTime()),
                        "Kafka Source");

        // keyBy userId and productionId
        // Notes, only events with the same key will be processd to see if there is a match
        KeyedStream<Event, Tuple2<Integer, Integer>> keyedStream =
                source.assignTimestampsAndWatermarks(
                        WatermarkStrategy.<Event>forGenerator(ctx -> new EventBoundedOutOfOrdernessWatermarks(Duration.ofSeconds(5)))
                ).keyBy(new KeySelector<Event, Tuple2<Integer, Integer>>() {
                    @Override
                    public Tuple2<Integer, Integer> getKey(Event value) throws Exception {
                        return Tuple2.of(value.getId(), value.getProductionId());
                    }
                });

        // Dynamic CEP patterns
        SingleOutputStreamOperator<String> output =
                CEP.dynamicPatterns(
                        keyedStream,
                        new JDBCPeriodicPatternProcessorDiscovererFactory<>(
                                params.get(JDBC_URL_ARG),
                                JDBC_DRIVE,
                                params.get(TABLE_NAME_ARG),
                                null,
                                Long.parseLong(params.get(JDBC_INTERVAL_MILLIS_ARG))),
                        Boolean.parseBoolean(params.get(USING_EVENT_TIME)) ? TimeBehaviour.EventTime : TimeBehaviour.ProcessingTime,
                        TypeInformation.of(new TypeHint<String>() {}));

        output.print();

        // Sink to MySQL
        SingleOutputStreamOperator<Match_results> sinkMySQL = output.process(new ProcessFunction<String, Match_results>() {
            @Override
            public void processElement(String value, Context ctx, Collector<Match_results> out) throws Exception {
                Match_results results = parseOutput(value);
                if (results != null) {
                    out.collect(results);
                } else {
                    System.out.println("Parsing failure: " + value);
                }
            }
        });

        sinkMySQL.addSink(JdbcSink.sink(
                "INSERT IGNORE  INTO match_results (rule_id, rule_version, user_id, user_name, production_id)  VALUES (?, ?, ?, ?, ?)",
                new JdbcStatementBuilder<Match_results>() {
                    @Override
                    public void accept(PreparedStatement ps, Match_results results) throws SQLException {
                        if (results== null){
                            System.out.println("Event object is not found !");
                        }else {
                            ps.setInt(1, results.getRule_id());
                            ps.setInt(2, results.getRule_version());
                            ps.setInt(3, results.getEvent_id());
                            ps.setString(4, results.getEvent_name());
                            ps.setInt(5, results.getProduction_id());
                        }
                    }
                },
                new JdbcExecutionOptions.Builder()
                        .withBatchSize(5)    // Batch size
                        .withBatchIntervalMs(2000) // Batch interval
                        .build(),
                new JdbcConnectionOptions.JdbcConnectionOptionsBuilder()
                        .withUrl(params.get(JDBC_URL_ARG))
                        .withDriverName(JDBC_DRIVE)
                        .build()
        )).name("Sink MySQL");
        env.execute("CEP Demo");
    }


}

