/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.rest.credentials;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import java.io.File;
import java.time.Instant;
import java.util.Map;
import java.util.Optional;
import org.apache.iceberg.rest.util.FileReadUtils;
import org.apache.iceberg.rest.util.TokenCacheManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.auth.credentials.AwsCredentials;
import software.amazon.awssdk.auth.credentials.AwsCredentialsProvider;
import software.amazon.awssdk.auth.credentials.AwsSessionCredentials;

public class DlfLocalFileCredentialsProvider
implements AwsCredentialsProvider {
    private static final Logger LOG = LoggerFactory.getLogger(DlfLocalFileCredentialsProvider.class);
    private final String tokenFilePath;
    private final Map<String, String> properties;

    DlfLocalFileCredentialsProvider(Map<String, String> properties) {
        this.properties = (Map)Preconditions.checkNotNull(properties, (Object)"properties cannot be null");
        Preconditions.checkArgument((properties.get("dlf.token-path") != null ? 1 : 0) != 0, (Object)"tokenFilePath value cannot be null");
        this.tokenFilePath = properties.get("dlf.token-path");
    }

    public static DlfLocalFileCredentialsProvider create(Map<String, String> properties) {
        return new DlfLocalFileCredentialsProvider(properties);
    }

    public AwsCredentials resolveCredentials() {
        AwsSessionCredentials credentials = this.getCredentialsFromCache();
        if (credentials == null) {
            credentials = this.updateSessionCredentials();
        }
        credentials.expirationTime().ifPresent(t -> LOG.debug("Using STS credentials with expiration time of " + t));
        return credentials;
    }

    protected static AwsSessionCredentials readToken(String tokenFilePath) {
        Exception lastException = null;
        for (int retry = 1; retry <= 5; ++retry) {
            try {
                String tokenStr = FileReadUtils.readFileUtf8(new File(tokenFilePath));
                ObjectMapper mapper = new ObjectMapper();
                JsonNode root = mapper.readTree(tokenStr);
                String accessKeyId = root.get("AccessKeyId").asText();
                String accessKeySecret = root.get("AccessKeySecret").asText();
                String securityToken = root.get("SecurityToken").asText();
                String expiration = root.get("Expiration").asText();
                return AwsSessionCredentials.builder().accessKeyId(accessKeyId).secretAccessKey(accessKeySecret).sessionToken(securityToken).expirationTime(Instant.parse(expiration)).build();
            }
            catch (Exception e) {
                lastException = e;
                try {
                    Thread.sleep((long)retry * 1000L);
                    continue;
                }
                catch (InterruptedException e2) {
                    Thread.currentThread().interrupt();
                    throw new RuntimeException(e2);
                }
            }
        }
        throw new RuntimeException(lastException);
    }

    private AwsSessionCredentials getCredentialsFromCache() {
        Optional<AwsSessionCredentials> credentials = this.getMetaToken(this.tokenFilePath);
        if (credentials.isEmpty()) {
            return null;
        }
        if (credentials.get().expirationTime().isEmpty()) {
            return null;
        }
        Instant expireTime = (Instant)credentials.get().expirationTime().get();
        if (expireTime == null) {
            return null;
        }
        Instant actualTokenExpiration = Instant.now().plusMillis(3600000L);
        if (actualTokenExpiration.isAfter(expireTime)) {
            return null;
        }
        return credentials.get();
    }

    private AwsSessionCredentials updateSessionCredentials() {
        AwsSessionCredentials credentials = DlfLocalFileCredentialsProvider.readToken(this.tokenFilePath);
        credentials.expirationTime().orElseThrow(() -> new IllegalStateException("Sourced credentials have no expiration value"));
        DlfLocalFileCredentialsProvider.updateMetaToken(this.tokenFilePath, credentials);
        return credentials;
    }

    private Optional<AwsSessionCredentials> getMetaToken(String path) {
        String cacheKey = TokenCacheManager.buildLocalFileTokenCacheKey(path);
        AwsSessionCredentials credentials = (AwsSessionCredentials)TokenCacheManager.getInstance().getCache().getIfPresent((Object)cacheKey);
        if (credentials != null) {
            return Optional.of(credentials);
        }
        return Optional.empty();
    }

    private static synchronized void updateMetaToken(String path, AwsSessionCredentials credentials) {
        String cacheKey = TokenCacheManager.buildLocalFileTokenCacheKey(path);
        TokenCacheManager.getInstance().getCache().put((Object)cacheKey, (Object)credentials);
    }
}

