/*
 * Decompiled with CFR 0.152.
 */
package org.apache.iceberg.rest.credentials;

import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import java.time.Instant;
import java.util.Map;
import java.util.Optional;
import org.apache.http.HttpEntity;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpGet;
import org.apache.http.client.methods.HttpUriRequest;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;
import org.apache.http.impl.client.HttpClients;
import org.apache.http.util.EntityUtils;
import org.apache.iceberg.rest.util.TokenCacheManager;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import software.amazon.awssdk.auth.credentials.AwsCredentials;
import software.amazon.awssdk.auth.credentials.AwsCredentialsProvider;
import software.amazon.awssdk.auth.credentials.AwsSessionCredentials;

public class DlfEcsTokenCredentialsProvider
implements AwsCredentialsProvider {
    private static final Logger LOG = LoggerFactory.getLogger(DlfEcsTokenCredentialsProvider.class);
    private final Map<String, String> properties;
    private String ecsMetadataURL;
    private String roleName;
    private final CloseableHttpClient httpClient;

    DlfEcsTokenCredentialsProvider(Map<String, String> properties) {
        this.properties = (Map)Preconditions.checkNotNull(properties, (Object)"properties cannot be null");
        this.ecsMetadataURL = properties.get("dlf.token-ecs-metadata-url");
        if (this.ecsMetadataURL == null || this.ecsMetadataURL.isEmpty()) {
            this.ecsMetadataURL = "http://100.100.100.200/latest/meta-data/Ram/security-credentials/";
        }
        this.roleName = properties.get("dlf.token-ecs-role-name");
        HttpClientBuilder clientBuilder = HttpClients.custom();
        this.httpClient = clientBuilder.build();
    }

    public static DlfEcsTokenCredentialsProvider create(Map<String, String> properties) {
        return new DlfEcsTokenCredentialsProvider(properties);
    }

    public AwsCredentials resolveCredentials() {
        AwsSessionCredentials credentials = this.getCredentialsFromCache();
        if (credentials == null) {
            credentials = this.updateSessionCredentials();
        }
        credentials.expirationTime().ifPresent(t -> LOG.debug("Using STS credentials with expiration time of " + t));
        return credentials;
    }

    private String getRole(String url) {
        return this.getResponseBody(url);
    }

    private AwsSessionCredentials getCredentialsFromCache() {
        Optional<AwsSessionCredentials> credentials = this.getMetaToken(this.ecsMetadataURL);
        if (credentials.isEmpty()) {
            return null;
        }
        if (credentials.get().expirationTime().isEmpty()) {
            return null;
        }
        Instant expireTime = (Instant)credentials.get().expirationTime().get();
        if (expireTime == null) {
            return null;
        }
        Instant actualTokenExpiration = Instant.now().plusMillis(3600000L);
        if (actualTokenExpiration.isAfter(expireTime)) {
            return null;
        }
        return credentials.get();
    }

    private AwsSessionCredentials getToken(String url) {
        String tokenStr = this.getResponseBody(url);
        try {
            ObjectMapper mapper = new ObjectMapper();
            JsonNode root = mapper.readTree(tokenStr);
            String accessKeyId = root.get("AccessKeyId").asText();
            String accessKeySecret = root.get("AccessKeySecret").asText();
            String securityToken = root.get("SecurityToken").asText();
            String expiration = root.get("Expiration").asText();
            return AwsSessionCredentials.builder().accessKeyId(accessKeyId).secretAccessKey(accessKeySecret).sessionToken(securityToken).expirationTime(Instant.parse(expiration)).build();
        }
        catch (Exception e) {
            LOG.error("Error parsing token", (Throwable)e);
            throw new RuntimeException(e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private String getResponseBody(String url) {
        long startTime = System.currentTimeMillis();
        HttpGet request = new HttpGet(url);
        try (CloseableHttpResponse response = this.httpClient.execute((HttpUriRequest)request);){
            int statusCode = response.getStatusLine().getStatusCode();
            if (statusCode != 200) throw new RuntimeException("Failed to fetch token, HTTP status: " + statusCode);
            LOG.debug("Get response success, url : {}, cost : {} ms", (Object)url, (Object)(System.currentTimeMillis() - startTime));
            String string = EntityUtils.toString((HttpEntity)response.getEntity());
            return string;
        }
        catch (Exception e) {
            LOG.error("Error fetching response", (Object)e.getMessage());
            throw new RuntimeException(e);
        }
    }

    private AwsSessionCredentials updateSessionCredentials() {
        if (this.roleName == null) {
            this.roleName = this.getRole(this.ecsMetadataURL);
        }
        String tokenUrl = this.ecsMetadataURL.endsWith("/") ? this.ecsMetadataURL + this.roleName : this.ecsMetadataURL + "/" + this.roleName;
        AwsSessionCredentials credentials = this.getToken(tokenUrl);
        credentials.expirationTime().ifPresent(t -> LOG.debug("Using STS credentials with expiration time of {}", t));
        DlfEcsTokenCredentialsProvider.updateMetaToken(this.ecsMetadataURL, credentials);
        return credentials;
    }

    private Optional<AwsSessionCredentials> getMetaToken(String path) {
        String cacheKey = TokenCacheManager.buildEcsTokenCacheKey(path);
        AwsSessionCredentials credentials = (AwsSessionCredentials)TokenCacheManager.getInstance().getCache().getIfPresent((Object)cacheKey);
        if (credentials != null) {
            return Optional.of(credentials);
        }
        return Optional.empty();
    }

    private static synchronized void updateMetaToken(String path, AwsSessionCredentials credentials) {
        String cacheKey = TokenCacheManager.buildEcsTokenCacheKey(path);
        TokenCacheManager.getInstance().getCache().put((Object)cacheKey, (Object)credentials);
    }
}

