import logging
import cbor
import json

def handler(event, context):
    logger = logging.getLogger()
    logger.info("Begin to handle event")

    # This allows easy testing in the console (JSON) and production use (CBOR) without code changes.
    records = None
    try:
        # Try decoding as CBOR first (standard Tablestore trigger format)
        records = cbor.loads(event)
    except (ValueError, TypeError):
        try:
            # If CBOR fails, try decoding as JSON (common for console testing)
            records = json.loads(event)
        except Exception as e:
            logger.error("Failed to decode event. It must be valid JSON or CBOR.")
            logger.error(e)
            return 'Error'

    if not records or 'Records' not in records:
        logger.warning("Event contains no records.")
        return 'OK'

    for record in records['Records']:
        logger.info("--------------------------------------------------")
        logger.info("Event Type: %s", record.get('Type'))
        
        # Log Record Metadata
        if 'Info' in record:
            logger.info("Timestamp: %s", record['Info'].get('Timestamp'))

        # Log Primary Keys
        logger.info("Primary Keys:")
        pk_list = record.get('PrimaryKey', [])
        for pk in pk_list:
            name = pk.get('ColumnName')
            value = pk.get('Value')
            logger.info("  Name: %s, Value: %s", name, value)

        # Log Attribute Columns
        logger.info("Attribute Columns:")
        columns_list = record.get('Columns', [])
        for col in columns_list:
            op_type = col.get('Type') # e.g., Put, Delete, DeleteAllVersions
            name = col.get('ColumnName')
            value = col.get('Value')
            timestamp = col.get('Timestamp')
            logger.info("  Type: %s, Name: %s, Value: %s, Timestamp: %s", 
                        op_type, name, value, timestamp)
        
        logger.info("--------------------------------------------------")

    return 'OK'




