/*
 * Copyright (c) 2024, Alibaba Cloud;
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.aliyun.migration.client;

import com.alibaba.fastjson2.JSONObject;
import com.aliyun.migration.api.context.TaskContext;
import com.aliyun.migration.api.plugin.Writer;
import com.aliyun.migration.teleport.common.conf.workflow.DefaultPropConf;
import com.aliyun.migration.teleport.common.element.meta.Datasource;
import com.aliyun.migration.workflow.migration.common.config.BwmSaasConfiguration;
import com.aliyun.migration.workflow.migration.common.file.BwmPackageFileOperator.JsonType;
import com.aliyun.migration.workflow.migration.common.spi.PluginHandler;
import com.aliyun.migration.workflow.migration.common.utils.BwmConfiguration;
import com.aliyun.migration.workflow.migration.common.utils.BwmJsonUtil;
import com.aliyun.migration.workflow.migration.common.utils.cloud.OssCloudClient;
import com.google.common.base.Charsets;
import java.io.File;
import java.util.HashMap;
import java.util.Objects;
import java.util.Optional;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import org.springframework.util.CollectionUtils;

/**
 * 写命令行接口
 *
 * @author jiman
 * @since 2024/7/12
 */
@Slf4j
public class WriterCommand extends Command{

    private Writer writer;
    public WriterCommand(){
        addOption();
    }

    @Override
    public void getPlugin(TaskContext taskContext) {
        writer = PluginHandler.getWriter(taskContext.getWriterName());
        commandName = "writer";
    }

    @Override
    public void closePlugin() {
        Optional.ofNullable(writer).ifPresent(Writer::close);
    }

    @Override
    public TaskContext getTaskContext(CommandLine cmd) throws Exception {
        // -t 转换插件类型
        String writerType = cmd.getOptionValue("t");
        // -c 配置
        String configStr = JsonType.JSON_TYPE.value;
        if(cmd.hasOption("c")){
            String configPath = cmd.getOptionValue("c");
            File file = new File(configPath);
            if(file.exists()){
                configStr = FileUtils.readFileToString(file, Charsets.UTF_8.name());
            } else {
                log.error("配置文件不存在 {}",configPath);
            }
        }
        // 解析配置，从中获取数据源信息
        BwmConfiguration configration = BwmConfiguration.from(configStr);
        Datasource datasource;
        String sourceSchedule = configration.getString("schedule_datasource");
        if (!StringUtils.isEmpty(sourceSchedule)) {
            datasource = BwmJsonUtil.parseObject(sourceSchedule, Datasource.class);
            if (Objects.isNull(datasource)) {
                log.error("无法解析配置文件中的迁移源端数据源信息：{}", sourceSchedule);
                datasource = new Datasource();
            }
        } else {
            log.warn("配置项中未获取到源端调度数据源信息，默认使用空数据源");
            datasource = new Datasource();
        }
        if(CollectionUtils.isEmpty(datasource.getProperties())){
            datasource.setProperties(new HashMap<>());
        }
        // 文件路径传参设置 -f 输入文件路径 -o 输出文件路径
        if (cmd.hasOption("f")) {
            String sourceFilePath = cmd.getOptionValue("f");
            if (!StringUtils.isEmpty(sourceFilePath)) {
                datasource.getProperties().put(DefaultPropConf.SOURCE_FILE_PATH, sourceFilePath);
            }
        }
        if (cmd.hasOption("o")) {
            String outFilePath = cmd.getOptionValue("o");
            if (!StringUtils.isEmpty(outFilePath)) {
                datasource.getProperties().put(DefaultPropConf.EXPORT_FILE_PATH, outFilePath);
            }
        }
        // 填充TaskContext
        TaskContext taskContext = new TaskContext();
        taskContext.setWriterDatasource(datasource);
        taskContext.setWriterName(writerType);
        taskContext.setBwmConfiguration(configStr);
        taskContext.setSaasConfiguration(configration.getString("saas"));
        return taskContext;
    }

    @Override
    public void execLocal(TaskContext taskContext) throws Exception {
        String sourceFilePath = taskContext.getWriterDatasource().getProperties().get(DefaultPropConf.SOURCE_FILE_PATH);
        String exportFilePath = taskContext.getWriterDatasource().getProperties().get(DefaultPropConf.EXPORT_FILE_PATH);
        if (StringUtils.isEmpty(sourceFilePath)) {
            log.error("未指定待提交文件路径");
            return;
        } else if (!new File(sourceFilePath).exists()) {
            log.error("待提交文件不存在");
            return;
        }
        if (StringUtils.isEmpty(exportFilePath)) {
            exportFilePath = "/temp/export/WriterOutput.zip";
            log.warn("未指定输出文件路径，默认使用{}", exportFilePath);
            taskContext.getWriterDatasource().getProperties().put(DefaultPropConf.EXPORT_FILE_PATH, exportFilePath);
        }
        // 执行提交
        writer.init(taskContext);
        writer.write();
        writer.makeXlxStatisticOp();
        writer.export();
    }

    @Override
    public void execSaas(TaskContext taskContext) throws Exception {
        BwmSaasConfiguration bwmSaasConfiguration = JSONObject.parseObject(taskContext.getSaasConfiguration(), BwmSaasConfiguration.class);
        if (Objects.isNull(bwmSaasConfiguration.getLocal())) {
            log.info("SaaS配置中local配置为空, use default");
            bwmSaasConfiguration.getLocal().useDefault("");
            taskContext.setSaasConfiguration(JSONObject.toJSONString(bwmSaasConfiguration));
        }
        if (Objects.isNull(bwmSaasConfiguration.getOss())) {
            log.warn("SaaS配置中oss配置为空, 执行完成后将跳过上传");
        }
        String sourceFilePath = bwmSaasConfiguration.getLocal().getInputFileReadPath().getMainPackage();
        String exportFilePath = bwmSaasConfiguration.getMigrationPackageZipSavePath();
        if (ObjectUtils.isEmpty(taskContext.getWriterDatasource().getProperties())) {
            taskContext.getWriterDatasource().setProperties(new HashMap<>());
        }
        taskContext.getWriterDatasource().getProperties().put(DefaultPropConf.SOURCE_FILE_PATH, sourceFilePath);
        taskContext.getWriterDatasource().getProperties().put(DefaultPropConf.EXPORT_FILE_PATH, exportFilePath);
        execLocal(taskContext);
    }

    @Override
    public void uploadFileToOss(TaskContext taskContext) {
        if (StringUtils.isEmpty(taskContext.getSaasConfiguration())) {
            log.warn("未获取到SaaS配置，跳过上传");
            return;
        }
        log.info("开始上传文件到OSS");
        BwmSaasConfiguration bwmSaasConfiguration = JSONObject.parseObject(taskContext.getSaasConfiguration(), BwmSaasConfiguration.class);
        if (Objects.isNull(bwmSaasConfiguration.getLocal())) {
            log.warn("SaaS配置中local配置为空, 跳过上传");
            return;
        }
        if (Objects.isNull(bwmSaasConfiguration.getOss())) {
            log.warn("SaaS配置中oss配置为空, 跳过上传");
            return;
        }
        OssCloudClient ossCloudClient = new OssCloudClient(bwmSaasConfiguration.getOss().getEndpoint(), bwmSaasConfiguration.getOss().getAk(), bwmSaasConfiguration.getOss().getSk());
        ossCloudClient.uploadAndMergeFolder(bwmSaasConfiguration.getOss().getBucket(), bwmSaasConfiguration.getOss().getRootPath(), bwmSaasConfiguration.getLocal().getOutputFileSaveRootPath());
        log.info("上传文件到OSS完成");
        // 上传完成标志
        ossCloudClient.createFile(bwmSaasConfiguration.getOss().getBucket(), bwmSaasConfiguration.getMigrationInstanceStatusFileOssPath(), "{\"status\":\"finish\"}");
    }

    @Override
    public void addOption() {
        options.addOption(Option.builder("c").argName("conf").hasArg().desc("配置文件").required(true).build());
        options.addOption(Option.builder("f").argName("file").hasArg().desc("（可选）原始文件").required(false).build());
        options.addOption(Option.builder("o").argName("out").hasArg().desc("输出文件名称及路径").required(true).build());
        options.addOption(Option.builder("t").argName("type").hasArg().desc("写插件类型").required(true).build());
    }

    @Override
    public String getName() {
        return ClientContants.WRITER_COMMAND_NAME;
    }

    @Override
    public String desc() {
        return "通过指定配置完成调度任务的写入";
    }
}
