/*
 * Copyright (c) 2024, Alibaba Cloud;
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.aliyun.migration.client;

import com.aliyun.migration.api.WorkflowProjectPackage;
import com.aliyun.migration.api.context.TaskContext;
import com.aliyun.migration.api.plugin.Converter;
import com.aliyun.migration.api.plugin.Reader;
import com.aliyun.migration.api.plugin.Writer;
import com.aliyun.migration.teleport.common.element.meta.Datasource;
import com.aliyun.migration.workflow.migration.common.file.BwmPackageFileOperator.JsonType;
import com.aliyun.migration.workflow.migration.common.spi.PluginHandler;
import com.aliyun.migration.workflow.migration.common.utils.BwmConfiguration;
import com.aliyun.migrationx.common.utils.GsonUtils;
import com.google.common.base.Charsets;
import java.io.File;
import java.util.Optional;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;
import org.apache.commons.io.FileUtils;

/**
 * 一键迁移客户端
 *
 * @author jiman
 * @since 2024/7/22
 */
@Slf4j
public class MigrationCommand extends Command{

    public MigrationCommand(){
        addOption();
    }

    @Override
    public void exec(String[] args) throws Exception {
        CommandLine cmd = parse.parse(options,args);
        if(cmd.hasOption("help")) {
            usage();
            return;
        }
        String configStr = JsonType.JSON_TYPE.value;
        if(cmd.hasOption("c")){
            String configPath = cmd.getOptionValue("c");
            File file = new File(configPath);
            if(file.exists()){
                configStr = FileUtils.readFileToString(file, Charsets.UTF_8.name());
            } else {
                log.error("config file path not exists {}",configPath);
            }
        }
        BwmConfiguration bwmConfiguration = BwmConfiguration.from(configStr);
        BwmConfiguration readerConf = bwmConfiguration.getConfiguration("reader");
        BwmConfiguration transformerConf = bwmConfiguration.getConfiguration("transformer");
        BwmConfiguration writerConf = bwmConfiguration.getConfiguration("writer");
        String readerName = readerConf.getString("name");
        String sourceScheduleDatasource = readerConf.getString("schedule_datasource");
        Datasource srcDatasource = GsonUtils.fromJsonString(sourceScheduleDatasource,Datasource.class);
        Reader reader = PluginHandler.getReader(readerName);
        TaskContext taskContext = new TaskContext();
        taskContext.setReaderDatasource(srcDatasource);
        taskContext.setReaderName(readerName);
        taskContext.setBwmConfiguration(readerConf.getString("conf"));
        reader.init(taskContext);
        WorkflowProjectPackage workflowProjectPackage = reader.read();

        String writerName = writerConf.getString("name");
        String tgtScheduleDatasourceStr = writerConf.getString("schedule_datasource");
        Datasource tgtDatasource = GsonUtils.fromJsonString(tgtScheduleDatasourceStr,Datasource.class);

        String transformName = transformerConf.getString("name");
        Converter converter = PluginHandler.getConverter(transformName);
        TaskContext transTaskContext = new TaskContext();
//        transTaskContext.setBwmConfiguration(transformerConf.getString("conf"));
        transTaskContext.setBwmConfiguration(transformerConf.toJSON());
        transTaskContext.setReaderDatasource(srcDatasource);
        transTaskContext.setWriterDatasource(tgtDatasource);
        transTaskContext.setConverterName(transformName);
        transTaskContext.setWorkflowProjectPackage(workflowProjectPackage);
        converter.init(transTaskContext);
        WorkflowProjectPackage convertPackage = converter.convert();

        Writer writer = PluginHandler.getWriter(writerName);
        TaskContext writerTaskContext = new TaskContext();
        writerTaskContext.setBwmConfiguration(writerConf.getString("conf"));
        writerTaskContext.setWriterDatasource(tgtDatasource);
        writerTaskContext.setWriterName(writerName);
        writerTaskContext.setWorkflowProjectPackage(convertPackage);
        writer.init(writerTaskContext);
        writer.write();

        close(reader,converter,writer);
    }

    @Override
    public TaskContext getTaskContext(CommandLine cmd) throws Exception {
        return null;
    }

    @Override
    public void getPlugin(TaskContext taskContext) throws Exception {

    }

    @Override
    public void execLocal(TaskContext taskContext) throws Exception {

    }

    @Override
    public void execSaas(TaskContext taskContext) throws Exception {

    }

    @Override
    public void closePlugin() {

    }

    private void close(Reader reader,Converter converter,Writer writer){
        Optional.ofNullable(reader).ifPresent(Reader::close);
        Optional.ofNullable(converter).ifPresent(Converter::close);
        Optional.ofNullable(writer).ifPresent(Writer::close);
    }

    @Override
    public void addOption() {
        options.addOption(Option.builder("c").argName("conf").hasArg().desc("配置文件").required(true).build());
    }

    @Override
    public String getName() {
        return ClientContants.MIGRATION_COMMAND_NAME;
    }

    @Override
    public String desc() {
        return "通过配置完成一键任务上云";
    }
}
