/*
 * Copyright (c) 2024, Alibaba Cloud;
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.aliyun.migration.client;

import com.alibaba.fastjson2.JSONObject;
import com.aliyun.migration.api.WorkflowProjectPackage;
import com.aliyun.migration.api.context.TaskContext;
import com.aliyun.migration.api.plugin.Reader;
import com.aliyun.migration.teleport.common.conf.workflow.DefaultPropConf;
import com.aliyun.migration.teleport.common.element.meta.Datasource;
import com.aliyun.migration.teleport.common.element.meta.workflow.WorkflowProject;
import com.aliyun.migration.workflow.migration.common.config.BwmSaasConfiguration;
import com.aliyun.migration.workflow.migration.common.file.BwmPackageFileOperator.JsonType;
import com.aliyun.migration.workflow.migration.common.file.BwmPackageSrResultStatisticFileOperator;
import com.aliyun.migration.workflow.migration.common.file.BwmPackageSrSnapshotFileOperator;
import com.aliyun.migration.workflow.migration.common.spi.PluginHandler;
import com.aliyun.migration.workflow.migration.common.utils.BwmConfiguration;
import com.aliyun.migration.workflow.migration.common.utils.BwmJsonUtil;
import com.aliyun.migration.workflow.migration.common.utils.cloud.CallbackProperties;
import com.aliyun.migration.workflow.migration.common.utils.cloud.ExecCallbackMessage;
import com.aliyun.migration.workflow.migration.common.utils.cloud.MnsClient;
import com.aliyun.migration.workflow.migration.common.utils.cloud.OssCloudClient;
import com.google.common.base.Charsets;
import java.io.File;
import java.io.IOException;
import java.util.*;

import lombok.extern.slf4j.Slf4j;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;
import org.apache.commons.io.FileUtils;
import org.springframework.util.CollectionUtils;
import org.apache.commons.lang3.StringUtils;

/**
 * 读命令行接口
 *
 * @author jiman
 * @since 2024/7/12
 */
@Slf4j
public class ReaderCommand extends Command{

    private Reader reader;
    private List<String> projects;
    public ReaderCommand(){
        addOption();
    }

    @Override
    public void getPlugin(TaskContext taskContext) {
        reader = PluginHandler.getReader(taskContext.getReaderName());
        commandName = "reader";
    }

    @Override
    public void closePlugin() {
        Optional.ofNullable(reader).ifPresent(Reader::close);
    }

    @Override
    public TaskContext getTaskContext(CommandLine cmd) throws Exception {
        // -t 获取Reader插件名称
        String readerType = cmd.getOptionValue("t");
        // -c 获取Reader配置
        String configStr = JsonType.JSON_TYPE.value;
        if(cmd.hasOption("c")){
            String configPath = cmd.getOptionValue("c");
            File file = new File(configPath);
            if(file.exists()){
                configStr = FileUtils.readFileToString(file, Charsets.UTF_8.name());
            } else {
                log.error("配置文件不存在：{}",configPath);
            }
        }
        // 解析Reader配置，从中获取数据源信息
        BwmConfiguration configration = BwmConfiguration.from(configStr);
        Datasource datasource;
        String sourceSchedule = configration.getString("schedule_datasource");
        if (!StringUtils.isEmpty(sourceSchedule)) {
            datasource = BwmJsonUtil.parseObject(sourceSchedule, Datasource.class);
            if (Objects.isNull(datasource)) {
                log.error("无法解析配置文件中的迁移源端数据源信息：{}", sourceSchedule);
                datasource = new Datasource();
            }
        } else {
            log.warn("配置项中未获取到源端调度数据源信息，默认使用空数据源");
            datasource = new Datasource();
        }
        if(CollectionUtils.isEmpty(datasource.getProperties())){
            datasource.setProperties(new HashMap<>());
        }
        // 文件路径传参设置 -f 输入文件路径 -o 输出文件路径
        if (cmd.hasOption("f")) {
            String sourceFilePath = cmd.getOptionValue("f");
            if (!StringUtils.isEmpty(sourceFilePath)) {
                datasource.getProperties().put(DefaultPropConf.SOURCE_FILE_PATH, sourceFilePath);
            }
        }
        if (cmd.hasOption("o")) {
            String outFilePath = cmd.getOptionValue("o");
            if (!StringUtils.isEmpty(outFilePath)) {
                datasource.getProperties().put(DefaultPropConf.EXPORT_FILE_PATH, outFilePath);
            }
        }
        // 填充TaskContext
        TaskContext taskContext = new TaskContext();
        taskContext.setReaderDatasource(datasource);
        taskContext.setReaderName(readerType);
        taskContext.setBwmConfiguration(configration.getString("conf"));
        taskContext.setSaasConfiguration(configration.getString("saas"));
        return taskContext;
    }

    @Override
    public void execLocal(TaskContext taskContext) throws Exception {
        // 开始执行Reader
        String sourcePath = taskContext.getReaderDatasource().getProperties().get(DefaultPropConf.SOURCE_FILE_PATH);
        String exportPath = taskContext.getReaderDatasource().getProperties().get(DefaultPropConf.EXPORT_FILE_PATH);
        if (taskContext.getReaderDatasource().getProperties().containsKey(DefaultPropConf.OPERATOR) &&
                ((String) taskContext.getReaderDatasource().getProperties().get(DefaultPropConf.OPERATOR)).equalsIgnoreCase(Datasource.OperaterType.MANUAL.name())) {
            // 路径判空
            if (StringUtils.isEmpty(sourcePath) || (!(new File(sourcePath)).exists())) {
                log.error("离线模式，输入文件不存在：{}", sourcePath);
                throw new RuntimeException("离线模式，输入文件不存在：" + sourcePath);
            }
            if (StringUtils.isEmpty(exportPath)) {
                taskContext.getReaderDatasource().getProperties().put(DefaultPropConf.EXPORT_FILE_PATH, "/temp/export/ReaderOutput.zip");
            }
            // 离线模式固定单Project
            reader.init(taskContext);
            reader.read();
            reader.makeStatisticOp();
            reader.export();
        } else {
            // 在线模式获取Project列表并依次探查
            // 获取Project列表
            projects = new ArrayList<>();
            if (taskContext.getReaderDatasource().getProperties().containsKey("project")) {
                // 从配置项中获取Project列表
                projects = Arrays.asList(taskContext.getReaderDatasource().getProperties().get("project").split(","));
            } else {
                // 全量获取Project列表
                try {
                    reader.init(taskContext);
                    List<WorkflowProject> workflowProjectList = reader.getProjectList();
                    for (WorkflowProject workflowProject : workflowProjectList) {
                        projects.add(workflowProject.getProjectId());
                    }
                } catch (Exception e) {
                    log.warn("无法获取到project列表",e);
                }
            }
            // Reader执行
            if (projects.size() == 0) {
                // 路径判空
                if (StringUtils.isEmpty(sourcePath)) {
                    taskContext.getReaderDatasource().getProperties().put(DefaultPropConf.SOURCE_FILE_PATH, "/temp/source/SourceOutput.zip");
                }
                if (StringUtils.isEmpty(exportPath)) {
                    taskContext.getReaderDatasource().getProperties().put(DefaultPropConf.EXPORT_FILE_PATH, "/temp/export/ReaderOutput.zip");
                }
                // 无Project信息或单Project，尝试直接处理
                reader.init(taskContext);
                reader.read();
                reader.makeStatisticOp();
                reader.export();
            } else if (projects.size() == 1) {
                // 路径判空
                if (StringUtils.isEmpty(sourcePath)) {
                    sourcePath = "/temp/source";
                }
                if (StringUtils.isEmpty(exportPath)) {
                    exportPath = "/temp/export";
                }
                // 单Project
                if ((new File(sourcePath)).isDirectory()) {
                    sourcePath = sourcePath + "/" + projects.get(0) + ".zip";
                }
                if ((new File(exportPath)).isDirectory()) {
                    exportPath = exportPath + "/" + projects.get(0) + ".zip";
                }
                taskContext.getReaderDatasource().getProperties().put("project", projects.get(0));
                taskContext.getReaderDatasource().getProperties().put(DefaultPropConf.SOURCE_FILE_PATH, sourcePath);
                taskContext.getReaderDatasource().getProperties().put(DefaultPropConf.EXPORT_FILE_PATH, exportPath);
                reader.init(taskContext);
                reader.read();
                reader.makeStatisticOp();
                reader.export();
            } else {
                // 路径判空
                if (StringUtils.isEmpty(sourcePath)) {
                    sourcePath = "/temp/source";
                }
                if (StringUtils.isEmpty(exportPath)) {
                    exportPath = "/temp/export";
                }
                // 多Project
                if ((new File(sourcePath)).isFile()) {
                    sourcePath = new File(sourcePath).getParent();
                }
                if ((new File(exportPath)).isFile()) {
                    exportPath = new File(exportPath).getParent();
                }
                for (String project: projects) {
                    try {
                        log.info("开始获取project: {}, 输出至：{}", project, exportPath + "/" + project + ".zip");
                        taskContext.getReaderDatasource().getProperties().put("project", project);
                        taskContext.getReaderDatasource().getProperties().put(
                                DefaultPropConf.SOURCE_FILE_PATH, sourcePath + "/" + project + ".zip");
                        taskContext.getReaderDatasource().getProperties().put(
                                DefaultPropConf.EXPORT_FILE_PATH, exportPath + "/" + project + ".zip");
                        reader.init(taskContext);
                        reader.read();
                        reader.makeStatisticOp();
                        reader.export();
                    } catch (Exception e) {
                        log.warn("获取project: {} 失败", project, e);
                    }
                }
            }
        }
    }

    @Override
    public void execSaas(TaskContext taskContext) throws Exception {
        // 开始执行Reader
        BwmSaasConfiguration bwmSaasConfiguration = JSONObject.parseObject(taskContext.getSaasConfiguration(), BwmSaasConfiguration.class);
        List<WorkflowProjectPackage> pkgs = new ArrayList<>();
        if (Objects.isNull(bwmSaasConfiguration.getLocal())) {
            log.info("SaaS配置中local配置为空, use default");
            bwmSaasConfiguration.getLocal().useDefault("");
            taskContext.setSaasConfiguration(JSONObject.toJSONString(bwmSaasConfiguration));
        }
        if (Objects.isNull(bwmSaasConfiguration.getOss())) {
            log.warn("SaaS配置中oss配置为空, 执行完成后将跳过上传");
        }
        if (taskContext.getReaderDatasource().getProperties().containsKey(DefaultPropConf.OPERATOR) &&
                ((String) taskContext.getReaderDatasource().getProperties().get(DefaultPropConf.OPERATOR)).equalsIgnoreCase(Datasource.OperaterType.MANUAL.name())) {
            // 输入路径判空
            String sourcePath = bwmSaasConfiguration.getLocal().getInputFileReadPath().getMainPackage();
            if (StringUtils.isEmpty(sourcePath) || (!(new File(sourcePath)).exists())) {
                log.error("离线模式，输入文件不存在：{}", sourcePath);
                throw new RuntimeException("离线模式，输入文件不存在：" + sourcePath);
            }
            // 强行指定输出位置
            String exportPath = bwmSaasConfiguration.getDiscoveryPackageZipSavePath(null);
            taskContext.getReaderDatasource().getProperties().put(DefaultPropConf.EXPORT_FILE_PATH, exportPath);
            // 离线模式固定单Project
            reader.init(taskContext);
            WorkflowProjectPackage pkg = reader.read();
            pkgs.add(BwmJsonUtil.deepCopy(pkg, WorkflowProjectPackage.class));
            reader.makeStatisticOp();
            reader.export();
        } else {
            // 在线模式获取Project列表并依次探查
            // 获取Project列表
            projects = new ArrayList<>();
            if (taskContext.getReaderDatasource().getProperties().containsKey("project")) {
                // 从配置项中获取Project列表
                projects = Arrays.asList(taskContext.getReaderDatasource().getProperties().get("project").split(","));
            } else {
                // 全量获取Project列表
                try {
                    reader.init(taskContext);
                    List<WorkflowProject> workflowProjectList = reader.getProjectList();
                    for (WorkflowProject workflowProject : workflowProjectList) {
                        projects.add(workflowProject.getProjectId());
                    }
                } catch (Exception e) {
                    log.warn("无法获取到project列表",e);
                }
            }
            // Reader执行
            if (projects.size() == 0) {
                String sourcePath = bwmSaasConfiguration.getDiscoveryApiOutputPackageZipSavePath(null);
                String exportPath = bwmSaasConfiguration.getDiscoveryPackageZipSavePath(null);
                taskContext.getReaderDatasource().getProperties().put(DefaultPropConf.SOURCE_FILE_PATH, sourcePath);
                taskContext.getReaderDatasource().getProperties().put(DefaultPropConf.EXPORT_FILE_PATH, exportPath);
                // 无Project信息或单Project，尝试直接处理
                reader.init(taskContext);
                WorkflowProjectPackage pkg = reader.read();
                pkgs.add(BwmJsonUtil.deepCopy(pkg, WorkflowProjectPackage.class));
                reader.makeStatisticOp();
                reader.export();
            } else if (projects.size() == 1) {
                // 路径判空
                String sourcePath = bwmSaasConfiguration.getDiscoveryApiOutputPackageZipSavePath(projects.get(0));
                String exportPath = bwmSaasConfiguration.getDiscoveryPackageZipSavePath(projects.get(0));
                taskContext.getReaderDatasource().getProperties().put("project", projects.get(0));
                taskContext.getReaderDatasource().getProperties().put(DefaultPropConf.SOURCE_FILE_PATH, sourcePath);
                taskContext.getReaderDatasource().getProperties().put(DefaultPropConf.EXPORT_FILE_PATH, exportPath);
                reader.init(taskContext);
                WorkflowProjectPackage pkg = reader.read();
                pkgs.add(BwmJsonUtil.deepCopy(pkg, WorkflowProjectPackage.class));
                reader.makeStatisticOp();
                reader.export();
            } else {
                for (String project: projects) {
                    try {
                        String sourcePath = bwmSaasConfiguration.getDiscoveryApiOutputPackageZipSavePath(project);
                        String exportPath = bwmSaasConfiguration.getDiscoveryPackageZipSavePath(project);
                        log.info("开始获取project: {}, 输出至：{}", project, exportPath);
                        taskContext.getReaderDatasource().getProperties().put("project", project);
                        taskContext.getReaderDatasource().getProperties().put(DefaultPropConf.SOURCE_FILE_PATH, sourcePath);
                        taskContext.getReaderDatasource().getProperties().put(DefaultPropConf.EXPORT_FILE_PATH, exportPath);
                        reader.init(taskContext);
                        WorkflowProjectPackage pkg = reader.read();
                        pkgs.add(BwmJsonUtil.deepCopy(pkg, WorkflowProjectPackage.class));
                        reader.makeStatisticOp();
                        reader.export();
                    } catch (Exception e) {
                        log.warn("获取project: {} 失败", project, e);
                    }
                }
            }
        }
        saveSaasInstanceOverviewSrJsonFile(pkgs, bwmSaasConfiguration);
    }

    public void saveSaasInstanceOverviewSrJsonFile(List<WorkflowProjectPackage> pkgs, BwmSaasConfiguration bwmSaasConfiguration) {
        BwmPackageSrResultStatisticFileOperator.writeDiscoverySrResultMappingOverviewFile(pkgs, bwmSaasConfiguration);
    }

    @Override
    public void copyLogFile(TaskContext taskContext) {
        BwmSaasConfiguration bwmSaasConfiguration = JSONObject.parseObject(taskContext.getSaasConfiguration(), BwmSaasConfiguration.class);
        String logFilePath = bwmSaasConfiguration.getDiscoveryLogFileSavePath(null);
        File systemLogFolder = new File("./logs");
        try {
            FileUtils.copyDirectory(systemLogFolder, new File(logFilePath));
        } catch (IOException e) {
            log.warn("拷贝日志文件失败", e);
        }
    }

    @Override
    public void uploadFileToOss(TaskContext taskContext) {
        if (StringUtils.isEmpty(taskContext.getSaasConfiguration())) {
            log.warn("未获取到SaaS配置，跳过上传");
            return;
        }
        log.info("开始上传文件到OSS");
        BwmSaasConfiguration bwmSaasConfiguration = JSONObject.parseObject(taskContext.getSaasConfiguration(), BwmSaasConfiguration.class);
        if (Objects.isNull(bwmSaasConfiguration.getLocal())) {
            log.warn("SaaS配置中local配置为空, 跳过上传");
            return;
        }
        if (Objects.isNull(bwmSaasConfiguration.getOss())) {
            log.warn("SaaS配置中oss配置为空, 跳过上传");
            return;
        }
        OssCloudClient ossCloudClient = new OssCloudClient(bwmSaasConfiguration.getOss().getEndpoint(), bwmSaasConfiguration.getOss().getAk(), bwmSaasConfiguration.getOss().getSk());
        ossCloudClient.uploadAndMergeFolder(bwmSaasConfiguration.getOss().getBucket(), bwmSaasConfiguration.getOss().getRootPath(), bwmSaasConfiguration.getLocal().getOutputFileSaveRootPath());
        log.info("上传文件到OSS完成");
        // 上传完成标志
        ossCloudClient.createFile(bwmSaasConfiguration.getOss().getBucket(), bwmSaasConfiguration.getDiscoveryInstanceStatusFileOssPath(), "{\"status\":\"finish\"}");
    }

    @Override
    public void sendMns(BwmSaasConfiguration bwmSaasConfiguration) {
        if (bwmSaasConfiguration == null) {
            log.warn("未获取到SaaS配置，跳过发送MNS");
            return;
        }
        CallbackProperties properties = new CallbackProperties();
        properties.setAccessKeyId(bwmSaasConfiguration.getMns().getAk());
        properties.setAccessKeySecret(bwmSaasConfiguration.getMns().getSk());
        properties.setEndpoint(bwmSaasConfiguration.getMns().getEndpoint());
        properties.setTopicName(bwmSaasConfiguration.getMns().getTopicName());
        MnsClient mnsClient = new MnsClient(properties);
        ExecCallbackMessage callbackMessage = new ExecCallbackMessage();
        callbackMessage.setTenantUuid(bwmSaasConfiguration.getMeta().getTenantUuid());
        callbackMessage.setTaskUuid(bwmSaasConfiguration.getMeta().getTaskUuid());
        callbackMessage.setInstanceChainUuid(bwmSaasConfiguration.getMeta().getInstanceChainUuid());
        callbackMessage.setInstanceUuid(bwmSaasConfiguration.getMeta().getInstanceUuid());
        callbackMessage.setMetaVersion(bwmSaasConfiguration.getMeta().getMetaVersion());
        callbackMessage.setScheduleDatasourceUuid(bwmSaasConfiguration.getMeta().getScheduleDatasourceUuid());
        callbackMessage.setInputMetaVersion(bwmSaasConfiguration.getMeta().getInputMetaVersion());
        callbackMessage.setIsSuccess(true);
        callbackMessage.setMsgTime(callbackMessage.ofBizDate());
        callbackMessage.setErrorMsg(null);
        callbackMessage.setStageType(bwmSaasConfiguration.getStageType().name());
        callbackMessage.setProjectIds(projects);
        mnsClient.sendMessage(callbackMessage);
        mnsClient.close();
    }


    @Override
    public void addOption() {
        options.addOption(Option.builder("c").argName("conf").hasArg().desc("配置文件").required(false).build());
        options.addOption(Option.builder("f").argName("file").hasArg().desc("（可选）原始文件").required(false).build());
        options.addOption(Option.builder("o").argName("out").hasArg().desc("输出文件名称及路径").required(false).build());
        options.addOption(Option.builder("t").argName("type").hasArg().desc("读插件类型").required(true).build());
    }

    @Override
    public String getName() {
        return ClientContants.READER_COMMAND_NAME;
    }

    @Override
    public String desc() {
        return "通过指定配置完成调度的解析和读取";
    }
}
