/*
 * Copyright (c) 2024, Alibaba Cloud;
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.aliyun.migration.client;

import com.alibaba.fastjson2.JSONObject;
import com.aliyun.migration.api.constant.BwmCommonConstant;
import com.aliyun.migration.api.context.TaskContext;
import com.aliyun.migration.api.plugin.Converter;
import com.aliyun.migration.api.plugin.Reader;
import com.aliyun.migration.api.plugin.Writer;
import com.aliyun.migration.teleport.common.conf.workflow.DefaultPropConf;
import com.aliyun.migration.teleport.common.element.meta.Datasource;
import com.aliyun.migration.workflow.migration.common.config.BwmSaasConfiguration;
import com.aliyun.migration.workflow.migration.common.file.BwmPackageFileOperator;
import com.aliyun.migration.workflow.migration.common.file.BwmPackageFileOperator.JsonType;
import com.aliyun.migration.workflow.migration.common.spi.PluginHandler;
import com.aliyun.migration.workflow.migration.common.utils.BwmConfiguration;
import com.aliyun.migration.workflow.migration.common.utils.cloud.OssCloudClient;
import com.aliyun.migrationx.common.utils.GsonUtils;
import org.apache.commons.lang3.ObjectUtils;
import org.apache.commons.lang3.StringUtils;
import com.google.common.base.Charsets;
import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.Option;
import org.apache.commons.io.FileUtils;
import org.springframework.util.CollectionUtils;

/**
 * 转换命令行接口
 *
 * @author jiman
 * @since 2024/7/12
 */
@Slf4j
public class ConverterCommand extends Command{

    private Converter converter;
    public ConverterCommand(){
        addOption();
    }

    @Override
    public void getPlugin(TaskContext taskContext) {
        converter = PluginHandler.getConverter(taskContext.getConverterName());
        commandName = "converter";
    }

    @Override
    public void closePlugin() {
        Optional.ofNullable(converter).ifPresent(Converter::close);
    }

    @Override
    public TaskContext getTaskContext(CommandLine cmd) throws Exception {
        // -t 转换插件类型
        String convertType = cmd.getOptionValue("t");
        // -c 配置
        String configStr = JsonType.JSON_TYPE.value;
        if(cmd.hasOption("c")){
            String configPath = cmd.getOptionValue("c");
            File file = new File(configPath);
            if(file.exists()){
                configStr = FileUtils.readFileToString(file, Charsets.UTF_8.name());
            } else {
                log.error("配置文件不存在 {}",configPath);
            }
        }
        // 解析配置，从中获取数据源信息
        BwmConfiguration Configuration = BwmConfiguration.from(configStr);
        String sourceSchedule = Configuration.getString("schedule_datasource");
        Datasource sourceDatasource;
        if (!StringUtils.isEmpty(sourceSchedule)) {
            sourceDatasource = GsonUtils.fromJsonString(sourceSchedule,Datasource.class);
            if (Objects.isNull(sourceDatasource)) {
                log.error("无法解析配置文件中的迁移源端数据源信息：{}", sourceSchedule);
                sourceDatasource = new Datasource();
            }
        } else {
            log.warn("配置项中未获取到源端调度数据源信息，默认使用空数据源");
            sourceDatasource = new Datasource();
        }
        if(CollectionUtils.isEmpty(sourceDatasource.getProperties())){
            sourceDatasource.setProperties(new HashMap<>());
        }
        String targetSchedule = Configuration.getString("target_schedule_datasource");
        Datasource targetDatasource;
        if (!StringUtils.isEmpty(targetSchedule)) {
            targetDatasource = GsonUtils.fromJsonString(targetSchedule,Datasource.class);
            if (Objects.isNull(targetDatasource)) {
                log.error("无法解析配置文件中的迁移目标端数据源信息：{}", targetSchedule);
                targetDatasource = new Datasource();
            }
        } else {
            log.warn("配置项中未获取到目标端调度数据源信息，默认使用空数据源");
            targetDatasource = new Datasource();
        }
        if(CollectionUtils.isEmpty(targetDatasource.getProperties())){
            targetDatasource.setProperties(new HashMap<>());
        }
        // 文件路径传参设置 -m ApiOutput文件路径 -f ReaderOutput文件路径 -o ConverterOutput输出文件路径
        if (cmd.hasOption("m")) {
            String originalFilePath = cmd.getOptionValue("m");
            if (!StringUtils.isEmpty(originalFilePath)) {
                sourceDatasource.getProperties().put(DefaultPropConf.SOURCE_FILE_PATH, originalFilePath);
            }
        }
        if (cmd.hasOption("f")) {
            String sourceFilePath = cmd.getOptionValue("f");
            if (!StringUtils.isEmpty(sourceFilePath)) {
                sourceDatasource.getProperties().put(DefaultPropConf.EXPORT_FILE_PATH, sourceFilePath);
            }
        }
        if (cmd.hasOption("o")) {
            String outputFilePath = cmd.getOptionValue("o");
            if (!StringUtils.isEmpty(outputFilePath)) {
                targetDatasource.getProperties().put(DefaultPropConf.SOURCE_FILE_PATH, outputFilePath);
            }
        }
        // 填充TaskContext
        TaskContext taskContext = new TaskContext();
        taskContext.setReaderDatasource(sourceDatasource);
        taskContext.setWriterDatasource(targetDatasource);
        taskContext.setConverterName(convertType);
        taskContext.setBwmConfiguration(configStr);
        taskContext.setSaasConfiguration(Configuration.getString("saas"));
        return taskContext;
    }

    @Override
    public void execLocal(TaskContext taskContext) throws IOException {
        String originalFilePath = taskContext.getReaderDatasource().getProperties().get(DefaultPropConf.SOURCE_FILE_PATH);
        String sourceFilePath = taskContext.getReaderDatasource().getProperties().get(DefaultPropConf.EXPORT_FILE_PATH);
        String exportFilePath = taskContext.getWriterDatasource().getProperties().get(DefaultPropConf.SOURCE_FILE_PATH);
        // 路径判空
        BwmConfiguration bwmConfiguration = BwmConfiguration.from(taskContext.getBwmConfiguration());
        Map<String, Object> selfConfMap = bwmConfiguration.getMap(BwmCommonConstant.CONVERT_CONF.SELF,new HashMap<>());
        Boolean isUseMx = (Boolean) selfConfMap.getOrDefault(BwmCommonConstant.CONVERT_CONF.IF_USE_MIGRATIONX_BEFORE,false);
        taskContext.setIsUserMx(isUseMx);
        if (isUseMx) {
            if (StringUtils.isEmpty(originalFilePath)) {
                log.error("使用MigrationX但未指定原始文件路径");
                return;
            } else if (!new File(originalFilePath).exists()) {
                log.error("使用MigrationX但原始文件不存在");
                return;
            }
        } else {
            if (StringUtils.isEmpty(sourceFilePath)) {
                log.error("未指定Reader输出文件路径");
                return;
            } else if (!new File(sourceFilePath).exists()) {
                log.error("Reader输出文件不存在");
                return;
            }
        }
        if (StringUtils.isEmpty(exportFilePath)) {
            exportFilePath = "/temp/export/ConvertorOutput.zip";
            log.warn("未指定输出文件路径，默认使用{}", exportFilePath);
            taskContext.getWriterDatasource().getProperties().put(DefaultPropConf.SOURCE_FILE_PATH, exportFilePath);
        }
        // 执行转换
        converter.init(taskContext);
        converter.convert();
        converter.makeXlxStatisticOp();
        converter.export();
    }

    @Override
    public void execSaas(TaskContext taskContext) throws IOException {
        BwmSaasConfiguration bwmSaasConfiguration = JSONObject.parseObject(taskContext.getSaasConfiguration(), BwmSaasConfiguration.class);
        if (Objects.isNull(bwmSaasConfiguration.getLocal())) {
            log.info("SaaS配置中local配置为空, use default");
            bwmSaasConfiguration.getLocal().useDefault("");
            taskContext.setSaasConfiguration(JSONObject.toJSONString(bwmSaasConfiguration));
        }
        if (Objects.isNull(bwmSaasConfiguration.getOss())) {
            log.warn("SaaS配置中oss配置为空, 执行完成后将跳过上传");
        }
        String originalFilePath = bwmSaasConfiguration.getLocal().getInputFileReadPath().getMxReaderOutputPackage();
        String sourceFilePath = bwmSaasConfiguration.getLocal().getInputFileReadPath().getMainPackage();
        String exportFilePath = bwmSaasConfiguration.getMigrationPackageZipSavePath();
        if (ObjectUtils.isEmpty(taskContext.getReaderDatasource().getProperties())) {
            taskContext.getReaderDatasource().setProperties(new HashMap<>());
        }
        if (ObjectUtils.isEmpty(taskContext.getWriterDatasource().getProperties())) {
            taskContext.getWriterDatasource().setProperties(new HashMap<>());
        }
        taskContext.getReaderDatasource().getProperties().put(DefaultPropConf.SOURCE_FILE_PATH, originalFilePath);
        taskContext.getReaderDatasource().getProperties().put(DefaultPropConf.EXPORT_FILE_PATH, sourceFilePath);
        taskContext.getWriterDatasource().getProperties().put(DefaultPropConf.SOURCE_FILE_PATH, exportFilePath);
        execLocal(taskContext);
    }

    @Override
    public void uploadFileToOss(TaskContext taskContext) {
        if (StringUtils.isEmpty(taskContext.getSaasConfiguration())) {
            log.warn("未获取到SaaS配置，跳过上传");
            return;
        }
        log.info("开始上传文件到OSS");
        BwmSaasConfiguration bwmSaasConfiguration = JSONObject.parseObject(taskContext.getSaasConfiguration(), BwmSaasConfiguration.class);
        if (Objects.isNull(bwmSaasConfiguration.getLocal())) {
            log.warn("SaaS配置中local配置为空, 跳过上传");
            return;
        }
        if (Objects.isNull(bwmSaasConfiguration.getOss())) {
            log.warn("SaaS配置中oss配置为空, 跳过上传");
            return;
        }
        OssCloudClient ossCloudClient = new OssCloudClient(bwmSaasConfiguration.getOss().getEndpoint(), bwmSaasConfiguration.getOss().getAk(), bwmSaasConfiguration.getOss().getSk());
        ossCloudClient.uploadAndMergeFolder(bwmSaasConfiguration.getOss().getBucket(), bwmSaasConfiguration.getOss().getRootPath(), bwmSaasConfiguration.getLocal().getOutputFileSaveRootPath());
        log.info("上传文件到OSS完成");
        // 上传完成标志
        ossCloudClient.createFile(bwmSaasConfiguration.getOss().getBucket(), bwmSaasConfiguration.getMigrationInstanceStatusFileOssPath(), "{\"status\":\"finish\"}");
    }

    @Override
    public void addOption() {
        options.addOption(Option.builder("c").argName("conf").hasArg().desc("配置文件").required(false).build());
        options.addOption(Option.builder("m").argName("file").hasArg().desc("原始文件").required(false).build());
        options.addOption(Option.builder("f").argName("file").hasArg().desc("Reader输出文件").required(false).build());
        options.addOption(Option.builder("o").argName("out").hasArg().desc("输出文件名称及路径").required(false).build());
        options.addOption(Option.builder("t").argName("type").hasArg().desc("转换插件类型").required(true).build());
    }

    @Override
    public String getName() {
        return ClientContants.CONVERTER_COMMAND_NAME;
    }

    @Override
    public String desc() {
        return "通过指定配置完成调度转换";
    }
}
