/*
 * Copyright (c) 2024, Alibaba Cloud;
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package com.aliyun.migration.client;

import com.alibaba.fastjson2.JSONObject;
import com.aliyun.migration.api.context.TaskContext;
import com.aliyun.migration.api.plugin.Reader;
import com.aliyun.migration.teleport.cloud.enums.CloudAccessType;
import com.aliyun.migration.workflow.migration.common.config.BwmSaasConfiguration;
import com.aliyun.migration.workflow.migration.common.spi.PluginHandler;
import com.aliyun.migration.workflow.migration.common.spi.RegisterOperator;
import com.aliyun.migration.workflow.migration.common.utils.BwmCommonUtil;
import java.io.File;
import java.io.IOException;
import java.util.Locale;
import java.util.Objects;
import java.util.Optional;

import com.aliyun.migration.workflow.migration.common.utils.cloud.CallbackProperties;
import com.aliyun.migration.workflow.migration.common.utils.cloud.ExecCallbackMessage;
import com.aliyun.migration.workflow.migration.common.utils.cloud.MnsClient;
import com.aliyun.migration.workflow.migration.common.utils.cloud.OssCloudClient;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.cli.*;
import org.apache.commons.collections4.CollectionUtils;
import org.apache.commons.io.FileUtils;
import org.apache.commons.lang3.StringUtils;

/**
 * 命令行接口
 *
 * @author jiman
 * @since 2024/7/12
 */
@Slf4j
public abstract class Command {

    protected HelpFormatter formatter = new HelpFormatter();
    protected CommandLineParser parse = new DefaultParser();
    protected Options options = new Options();
    protected String commandName;

    protected Command() {
        formatter.setOptionComparator((o1, o2) -> 0);
    }

    public void exec(String[] args) throws Exception {
        System.setProperty("file.encoding", "zh_CN.UTF-8");
        TaskContext taskContext = new TaskContext();
        try{
            CommandLine cmd = parse.parse(options,args);
            // help
            if(cmd.hasOption("help")){
                usage();
                return;
            }
            // 获取TaskContext
            taskContext = getTaskContext(cmd);
            // 下载输入包
            downloadFileFromOss(taskContext);
            // 执行
            getPlugin(taskContext);
            if (StringUtils.isEmpty(taskContext.getSaasConfiguration())) {
                // 非Saas
                execLocal(taskContext);
            } else {
                // Saas
                execSaas(taskContext);
                // 拷贝日志文件至指定目录
                copyLogFile(taskContext);
                // 上传文件至OSS
                uploadFileToOss(taskContext);
                // 发送MNS
                sendMns(taskContext);
            }
        } catch (Exception e){
            log.error("exec error，",e);
            usage();
            closePlugin();
            // 通过MNS发送错误信息
            if (StringUtils.isEmpty(taskContext.getSaasConfiguration())) {
                sendMns(taskContext, e.getMessage());
            }
            System.exit(1);
        } finally {
            closePlugin();
            log.info("exec finished!");
        }
        System.exit(0);
    }

    public abstract TaskContext getTaskContext(CommandLine cmd) throws Exception;

    public abstract void getPlugin(TaskContext taskContext) throws Exception;

    public abstract void execLocal(TaskContext taskContext) throws Exception;

    public abstract void execSaas(TaskContext taskContext) throws Exception;

    public abstract void closePlugin();

    public void copyLogFile(TaskContext taskContext) {
        BwmSaasConfiguration bwmSaasConfiguration = JSONObject.parseObject(taskContext.getSaasConfiguration(), BwmSaasConfiguration.class);
        String logFilePath = bwmSaasConfiguration.getMigrationLogFileSavePath();
        File systemLogFolder = new File("./logs");
        try {
            FileUtils.copyDirectory(systemLogFolder, new File(logFilePath));
        } catch (IOException e) {
            log.warn("拷贝日志文件失败", e);
        }
    }

    public void downloadFileFromOss(TaskContext taskContext) {
        if (StringUtils.isEmpty(taskContext.getSaasConfiguration())) {
            log.warn("未获取到SaaS配置，跳过下载");
            return;
        }
        log.info("开始下载文件");
        BwmSaasConfiguration bwmSaasConfiguration = JSONObject.parseObject(taskContext.getSaasConfiguration(), BwmSaasConfiguration.class);
        downloadFileFromOss(bwmSaasConfiguration);
    }

    public void downloadFileFromOss(BwmSaasConfiguration bwmSaasConfiguration) {
        if (Objects.isNull(bwmSaasConfiguration.getOss())) {
            log.warn("SaaS配置中oss配置为空, 跳过下载");
            return;
        }
        if (CollectionUtils.isEmpty(bwmSaasConfiguration.getOss().getDownloadPackageToLocal().keySet())) {
            log.warn("SaaS配置中oss配置的文件列表为空, 跳过下载");
            return;
        }
        OssCloudClient ossCloudClient = new OssCloudClient(bwmSaasConfiguration.getOss().getEndpoint(), bwmSaasConfiguration.getOss().getAk(), bwmSaasConfiguration.getOss().getSk());
        for (String fileName : bwmSaasConfiguration.getOss().getDownloadPackageToLocal().keySet()) {
            String objectName = bwmSaasConfiguration.getOss().getDownloadPackageToLocal().get(fileName);
            String localFilePath = fileName;
            if (objectName.startsWith("/")) {
                objectName = objectName.substring(1);
            }
            if (!(new File(localFilePath).getParentFile().exists())) {
                try {
                    FileUtils.forceMkdir(new File(localFilePath).getParentFile());
                } catch (IOException e) {
                    log.error("save oss file error", e);
                }
            }
            ossCloudClient.downloadFile(bwmSaasConfiguration.getOss().getBucket(), objectName, localFilePath);
        }
    }

    public void uploadFileToOss(TaskContext taskContext) {
        if (StringUtils.isEmpty(taskContext.getSaasConfiguration())) {
            log.warn("未获取到SaaS配置，跳过上传");
            return;
        }
        log.info("开始上传文件到OSS");
        BwmSaasConfiguration bwmSaasConfiguration = JSONObject.parseObject(taskContext.getSaasConfiguration(), BwmSaasConfiguration.class);
        if (Objects.isNull(bwmSaasConfiguration.getLocal())) {
            log.warn("SaaS配置中local配置为空, 跳过上传");
            return;
        }
        if (Objects.isNull(bwmSaasConfiguration.getOss())) {
            log.warn("SaaS配置中oss配置为空, 跳过上传");
            return;
        }
        OssCloudClient ossCloudClient = new OssCloudClient(bwmSaasConfiguration.getOss().getEndpoint(), bwmSaasConfiguration.getOss().getAk(), bwmSaasConfiguration.getOss().getSk());
        ossCloudClient.uploadAndMergeFolder(bwmSaasConfiguration.getOss().getBucket(), bwmSaasConfiguration.getOss().getRootPath(), bwmSaasConfiguration.getLocal().getOutputFileSaveRootPath());
        log.info("上传文件到OSS完成");
    }

    public void sendMns(TaskContext taskContext, String errorMsg) {
        if (StringUtils.isEmpty(taskContext.getSaasConfiguration())) {
            log.warn("未获取到SaaS配置，跳过发送MNS");
            return;
        }
        BwmSaasConfiguration bwmSaasConfiguration = JSONObject.parseObject(taskContext.getSaasConfiguration(), BwmSaasConfiguration.class);
        sendMns(bwmSaasConfiguration, errorMsg);
    }

    public void sendMns(TaskContext taskContext) {
        if (StringUtils.isEmpty(taskContext.getSaasConfiguration())) {
            log.warn("未获取到SaaS配置，跳过发送MNS");
            return;
        }
        BwmSaasConfiguration bwmSaasConfiguration = JSONObject.parseObject(taskContext.getSaasConfiguration(), BwmSaasConfiguration.class);
        sendMns(bwmSaasConfiguration);
    }

    public void sendMns(BwmSaasConfiguration bwmSaasConfiguration) {
        sendMns(bwmSaasConfiguration, null);
    }

    public void sendMns(BwmSaasConfiguration bwmSaasConfiguration, String errorMsg) {
        if (bwmSaasConfiguration == null) {
            log.warn("未获取到SaaS配置，跳过发送MNS");
            return;
        }
        CallbackProperties properties = new CallbackProperties();
        properties.setAccessKeyId(bwmSaasConfiguration.getMns().getAk());
        properties.setAccessKeySecret(bwmSaasConfiguration.getMns().getSk());
        properties.setEndpoint(bwmSaasConfiguration.getMns().getEndpoint());
        properties.setTopicName(bwmSaasConfiguration.getMns().getTopicName());
        MnsClient mnsClient = new MnsClient(properties);
        ExecCallbackMessage callbackMessage = new ExecCallbackMessage();
        callbackMessage.setTenantUuid(bwmSaasConfiguration.getMeta().getTenantUuid());
        callbackMessage.setTaskUuid(bwmSaasConfiguration.getMeta().getTaskUuid());
        callbackMessage.setInstanceChainUuid(bwmSaasConfiguration.getMeta().getInstanceChainUuid());
        callbackMessage.setInstanceUuid(bwmSaasConfiguration.getMeta().getInstanceUuid());
        callbackMessage.setMetaVersion(bwmSaasConfiguration.getMeta().getMetaVersion());
        callbackMessage.setScheduleDatasourceUuid(bwmSaasConfiguration.getMeta().getScheduleDatasourceUuid());
        callbackMessage.setInputMetaVersion(bwmSaasConfiguration.getMeta().getInputMetaVersion());
        if (StringUtils.isNotEmpty(errorMsg)) {
            callbackMessage.setIsSuccess(false);
            callbackMessage.setErrorMsg(errorMsg);
        } else {
            callbackMessage.setIsSuccess(true);
            callbackMessage.setErrorMsg(null);
        }
        callbackMessage.setMsgTime(callbackMessage.ofBizDate());
        callbackMessage.setStageType(bwmSaasConfiguration.getStageType().name());
        mnsClient.sendMessage(callbackMessage);
        mnsClient.close();
    }

    public abstract void addOption();

    public abstract String getName();

    public abstract String desc();

    public void usage(){
        formatter.printHelp(getName().toLowerCase(Locale.ROOT),desc()+"\n\n",options,"\n@几漫",true);
    }

    protected void init(){
        //加载插件
        String pluginLoadPath = BwmCommonUtil.getPluginLoadPath();
        try {
            FileUtils.forceMkdir(new File(pluginLoadPath));
        } catch (IOException e) {
            log.error("create plugin path error",e);
        }
        RegisterOperator.doRegister();
    }

}
