/*
 * Decompiled with CFR 0.152.
 */
package azkaban.executor;

import azkaban.executor.AlerterHolder;
import azkaban.executor.ExecutableFlow;
import azkaban.executor.ExecutionControllerUtils;
import azkaban.executor.ExecutionReference;
import azkaban.executor.Executor;
import azkaban.executor.ExecutorApiGateway;
import azkaban.executor.ExecutorLoader;
import azkaban.executor.ExecutorManagerException;
import azkaban.utils.Pair;
import azkaban.utils.Props;
import java.time.Duration;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import javax.inject.Inject;
import javax.inject.Singleton;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class ExecutorHealthChecker {
    private static final Logger logger = LoggerFactory.getLogger(ExecutorHealthChecker.class);
    private static final int DEFAULT_EXECUTOR_MAX_FAILURE_COUNT = 6;
    private static final Duration DEFAULT_EXECUTOR_HEALTHCHECK_INTERVAL = Duration.ofMinutes(5L);
    private final long healthCheckIntervalMin;
    private final int executorMaxFailureCount;
    private final List<String> alertEmails;
    private final ScheduledExecutorService scheduler;
    private final ExecutorLoader executorLoader;
    private final ExecutorApiGateway apiGateway;
    private final AlerterHolder alerterHolder;
    private final Map<Integer, Integer> executorFailureCount = new HashMap<Integer, Integer>();

    @Inject
    public ExecutorHealthChecker(Props azkProps, ExecutorLoader executorLoader, ExecutorApiGateway apiGateway, AlerterHolder alerterHolder) {
        this.healthCheckIntervalMin = azkProps.getLong("azkaban.executor.healthcheck.interval.min", DEFAULT_EXECUTOR_HEALTHCHECK_INTERVAL.toMinutes());
        this.executorMaxFailureCount = azkProps.getInt("azkaban.executor.max.failurecount", 6);
        this.alertEmails = azkProps.getStringList("azkaban.admin.alert.email");
        this.scheduler = Executors.newSingleThreadScheduledExecutor();
        this.executorLoader = executorLoader;
        this.apiGateway = apiGateway;
        this.alerterHolder = alerterHolder;
    }

    public void start() {
        logger.info("Starting executor health checker.");
        this.scheduler.scheduleAtFixedRate(() -> this.checkExecutorHealth(), 0L, this.healthCheckIntervalMin, TimeUnit.MINUTES);
    }

    public void shutdown() {
        this.scheduler.shutdown();
        try {
            if (!this.scheduler.awaitTermination(60L, TimeUnit.SECONDS)) {
                this.scheduler.shutdownNow();
            }
        }
        catch (InterruptedException ex) {
            this.scheduler.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    public void checkExecutorHealth() {
        Map<Optional<Executor>, List<ExecutableFlow>> exFlowMap = this.getFlowToExecutorMap();
        for (Map.Entry<Optional<Executor>, List<ExecutableFlow>> entry : exFlowMap.entrySet()) {
            Optional<Executor> executorOption = entry.getKey();
            if (!executorOption.isPresent()) {
                String finalizeReason = "Executor id of this execution doesn't exist.";
                for (ExecutableFlow flow : entry.getValue()) {
                    logger.warn(String.format("Finalizing execution %s, %s", flow.getExecutionId(), "Executor id of this execution doesn't exist."));
                    ExecutionControllerUtils.finalizeFlow(this.executorLoader, this.alerterHolder, flow, "Executor id of this execution doesn't exist.", null);
                }
                continue;
            }
            Executor executor = executorOption.get();
            try {
                Map<String, Object> results = this.apiGateway.callWithExecutionId(executor.getHost(), executor.getPort(), "ping", null, null, new Pair[0]);
                if (results == null || results.containsKey("error") || !results.containsKey("status") || !results.get("status").equals("alive")) {
                    throw new ExecutorManagerException("Status of executor " + executor.getId() + " is not alive.");
                }
                if (!this.executorFailureCount.containsKey(executor.getId())) continue;
                this.executorFailureCount.put(executor.getId(), 0);
            }
            catch (ExecutorManagerException e) {
                this.handleExecutorNotAliveCase(entry, executor, e);
            }
        }
    }

    private Map<Optional<Executor>, List<ExecutableFlow>> getFlowToExecutorMap() {
        HashMap<Optional<Executor>, List<ExecutableFlow>> exFlowMap = new HashMap<Optional<Executor>, List<ExecutableFlow>>();
        try {
            for (Pair<ExecutionReference, ExecutableFlow> runningFlow : this.executorLoader.fetchActiveFlows().values()) {
                Optional<Executor> executor = runningFlow.getFirst().getExecutor();
                List<ExecutableFlow> flows = exFlowMap.get(executor);
                if (flows == null) {
                    flows = new ArrayList<ExecutableFlow>();
                    exFlowMap.put(executor, flows);
                }
                flows.add(runningFlow.getSecond());
            }
        }
        catch (ExecutorManagerException e) {
            logger.error("Failed to get flow to executor map");
        }
        return exFlowMap;
    }

    private void handleExecutorNotAliveCase(Map.Entry<Optional<Executor>, List<ExecutableFlow>> entry, Executor executor, ExecutorManagerException e) {
        logger.error("Failed to get update from executor " + executor.getId(), (Throwable)e);
        this.executorFailureCount.put(executor.getId(), this.executorFailureCount.getOrDefault(executor.getId(), 0) + 1);
        if (this.executorFailureCount.get(executor.getId()) % this.executorMaxFailureCount == 0 && !this.alertEmails.isEmpty()) {
            entry.getValue().stream().forEach(flow -> flow.getExecutionOptions().setFailureEmails(this.alertEmails));
            logger.info(String.format("Executor failure count is %d. Sending alert emails to %s.", this.executorFailureCount.get(executor.getId()), this.alertEmails));
            this.alerterHolder.get("email").alertOnFailedUpdate(executor, entry.getValue(), e);
        }
    }
}

