from dashscope import Assistants, Messages, Runs, Threads
import json
# Import tool functions from tools.py.
from tools import ECS,Billing
# Import frontend display dependencies.
import gradio as gr
import ast
import dashscope

dashscope.base_http_api_url = 'https://dashscope-intl.aliyuncs.com/api/v1'
# A decision-making agent that decides which agents to use and their execution order.
PlannerAssistant = Assistants.create(
    # Because this agent plays an important role, choose a model with strong performance, such as qwen-plus. For a list of models, see https://www.alibabacloud.com/help/en/model-studio/getting-started/models.
    model="qwen-plus",  
    # Define the name of the agent.
    name='Flow Orchestration Bot',
    # Define the function description of the agent.
    description='You are the leader of a team of assistants. Based on user input, you need to decide the order in which to use these assistants.',
    # Define the instruction statement for the agent. The agent calls the tools and returns the results according to the instruction statement.
    instructions="""Your team has the following assistants. AliyunInfoAssistant: can query Alibaba Cloud ECS instance information in a user-specified region or query the user's Alibaba Cloud account balance. InstanceTypeDetailAssistant: can query detailed information about specified Alibaba Cloud ECS instance types, such as the number of vCPUs and memory size. It can query multiple instance types at once, so multiple calls are not required.
    ChatAssistant: is called if the user's question does not require the two assistants above. You need to determine the order in which to use these assistants based on the user's question. Your response must be in a list format, and no other information can be returned. For example: ["AliyunInfoAssistant", "AliyunInfoAssistant","InstanceTypeDetailAssistant"] or ["ChatAssistant"]. The elements in the list can only be the assistants mentioned above."""
)

# The function is to answer daily questions. For daily questions, you can use a model with a lower price as the base of the agent.
ChatAssistant = Assistants.create(
    # Because this agent does not have high requirements for the performance of the model, the lower-cost qwen-flash model is used. For a list of models, see https://www.alibabacloud.com/help/en/model-studio/getting-started/models.
    model="qwen-flash",
    name='Daily Q&A Bot',
    description='An intelligent assistant that answers user questions.',
    instructions='Please answer the user's questions politely.'
)

# The function is to query the resource information of Alibaba Cloud. Currently, it has two functions: ECS instance query and Alibaba Cloud account balance query.
AliyunInfoAssistant = Assistants.create(
    model="qwen-plus", # For a list of models, see https://www.alibabacloud.com/help/en/model-studio/getting-started/models.
    name='Alibaba Cloud Resource Query Bot',
    description='An intelligent assistant that calls tools based on user queries and returns the queried Alibaba Cloud resource results.',
    instructions='You are an intelligent assistant. You have two functions: Alibaba Cloud ECS instance information query and Alibaba Cloud account balance query. Please accurately determine which tool to call and answer the user's questions politely.',
    # Define the tools used by the agent. You can define one or more tools that the agent may use in the tools list based on your business scenario.
    tools=[
        {
            'type': 'function',
            'function': {
                # The name of the tool function. You can map the name to the function body through the function_mapper in the code below.
                'name': 'query_ecs_instance_info',
                # The description of the tool function.
                'description': 'This is very useful when you need to query Alibaba Cloud ECS instance information, such as instance ID, instance type, and fee information.',
                # The input parameters of the tool function.
                'parameters': {
                    'type': 'object',
                    'properties': {
                        # This tool requires the user to enter region information.
                        'RegionID': {
                            'type': 'str',
                            # The description of the parameter.
                            'description': 'The ID of the region where the instance resides. For example, use cn-hangzhou for China (Hangzhou), ap-southeast-1 for Singapore, or cn-beijing for China (Beijing).'
                        },
                    },
                    'required': ['RegionID']},
            }
        },
        {
            'type': 'function',
            'function': {
                'name': 'query_account_balance',
                # The description of the tool function.
                'description': 'This is very useful when you need to query Alibaba Cloud account information.',
                # The input parameters of the tool function. The balance query does not require input parameters, so it is empty.
                'parameters': {}
            }
        }
    ]
)

# The function is to query the detailed information of instance types through the RAG application created on the Alibaba Cloud Model Studio platform.
InstanceTypeDetailAssistant = Assistants.create(
    model="qwen-plus",  # For a list of models, see https://www.alibabacloud.com/help/en/model-studio/getting-started/models.
    name='ECS Instance Type Details Bot',
    description='An intelligent assistant that can accurately identify the mentioned instance types based on the user's input. It calls existing plug-in capabilities to introduce instance type information to users.',
    instructions='You are an intelligent assistant. You need to accurately identify and extract Alibaba Cloud instance type information from the user's input, such as [ecs.e-c1m1.large] or [ecs.u1-c1m4.xlarge,ecs.e-c1m1.large]. Input the instance type list into the tool to obtain their detailed information.',
    tools=[
        {
            'type': 'function',
            'function': {
                'name': 'get_ecs_instance_type_details',
                'description': 'Return the information of the specified ECS instance type queried by the customer.',
                'parameters': {
                    'type': 'object',
                    'properties': {
                        'InstanceType': {
                            'type': 'list',
                            'InstanceType': 'The instance type that the user wants to query. It can be a single instance type or multiple instance types, such as [ecs.e-c1m1.large] or [ecs.u1-c1m4.xlarge,ecs.e-c1m1.large].'
                        },
                    },
                    'required': ['InstanceType']},
            }
        }
    ]
)

# In a multi-agent scenario, define an agent for summarization. This agent provides a comprehensive and complete answer to the user's question based on the user's question and the reference information output by the previous agent.
SummaryAssistant = Assistants.create(
    model="qwen-plus",  # For a list of models, see https://www.alibabacloud.com/help/en/model-studio/getting-started/models.
    name='Summary Bot',
    description='An intelligent assistant that provides a comprehensive and complete answer to the user's question based on the user's question and reference information.',
    instructions='You are an intelligent assistant that provides a comprehensive and complete answer to the user's question based on the user's question and reference information.'
)

# Map the name of the tool function to the function body.
function_mapper = {
    "query_ecs_instance_info": ECS.query_source,
    "get_ecs_instance_type_details":ECS.call_agent_app,
    "query_account_balance":Billing.get_balance
}
# Map the name of the agent to the agent body.
assistant_mapper = {
    "ChatAssistant": ChatAssistant,
    "AliyunInfoAssistant":AliyunInfoAssistant,
    "InstanceTypeDetailAssistant":InstanceTypeDetailAssistant
}

# Get the response from a specified agent for a given message.
def get_agent_response(assistant, message=''):
    # Print the input agent information.
    print(f"Query: {message}")
    thread = Threads.create()
    message = Messages.create(thread.id, content=message)
    run = Runs.create(thread.id, assistant_id=assistant.id)
    run_status = Runs.wait(run.id, thread_id=thread.id)
    # If the response fails, "run failed" is printed.
    if run_status.status == 'failed':
        print('run failed:')
    # If a tool is needed to assist the model in outputting, perform the following process.
    if run_status.required_action:
        f = run_status.required_action.submit_tool_outputs.tool_calls[0].function
        # Get the function name.
        func_name = f['name']
        # Get the input parameters of the function.
        param = json.loads(f['arguments'])
        # Print the tool information.
        print("function is",f)
        # Map the function name to the function through function_mapper, and input the parameters into the tool function to get the output.
        if func_name in function_mapper:
            output = function_mapper[func_name](**param)
        else:    
            output = ""
        tool_outputs = [{
            'output':
                output
        }]
        run = Runs.submit_tool_outputs(run.id,
                                       thread_id=thread.id,
                                       tool_outputs=tool_outputs)
        run_status = Runs.wait(run.id, thread_id=thread.id)
    run_status = Runs.get(run.id, thread_id=thread.id)
    msgs = Messages.list(thread.id)
    # Return the output of the agent.
    return msgs['data'][0]['content'][0]['text']['value']

# Get the response from the multi-agent system. The input and output parameters must align with the Gradio frontend interface.
def get_multi_agent_response(query,history):
    # Handle empty input.
    if len(query) == 0:
        return "",history+[("","")],"",""
    # Get the agent execution order.
    assistant_order = get_agent_response(PlannerAssistant,query)
    try:
        order_stk = ast.literal_eval(assistant_order)
        cur_query = query
        Agent_Message = ""
        # Run the agents in sequence.
        for i in range(len(order_stk)):
            yield "----->".join(order_stk),history+[(query,"The multi-agent system is working...")],Agent_Message+'\n'+f"*{order_stk[i]}* is processing...",""
            cur_assistant = assistant_mapper[order_stk[i]]
            response = get_agent_response(cur_assistant,cur_query)
            Agent_Message += f"Response from *{order_stk[i]}*: {response}\n\n"
            yield "----->".join(order_stk),history+[(query,"The multi-agent system is working...")],Agent_Message,""
            # If the current agent is the last one in the sequence, generate the final response.
            if i == len(order_stk)-1:
                prompt = f"Based on the following information: {Agent_Message}, answer the user's question: {query}."
                multi_agent_response = get_agent_response(SummaryAssistant,prompt)
                yield "----->".join(order_stk),history+[(query,multi_agent_response)],Agent_Message,""
            # If not the last agent, pass the response to the next agent as context.
            else:
                # Add special identifiers to prevent the model from confusing context with the original question.
                cur_query = f"You can refer to the following information: {response}. You must provide a complete answer to the user's question. The question is: {query}."
    # Fallback policy: If the program fails, call ChatAssistant directly.
    except Exception as e:
        yield "ChatAssistant",[(query,get_agent_response(ChatAssistant,query))],"",""


# Frontend display interface
with gr.Blocks() as demo:
    # Display the title in the center of the interface.
    gr.HTML('<center><h1>Welcome to Alibaba Cloud Resource Query Bot</h1></center>')
    gr.HTML('<center><h3>Supported features include ECS instance query, balance query, and instance type detail query in specified regions. You can add the tools you need in tools.py and configure related agents in main.py.</h3></center>')
    with gr.Row():
        with gr.Column(scale=10):
            chatbot = gr.Chatbot(value=[["hello","Hello! I'm happy to help. What would you like to know about your Alibaba Cloud resources?"]],height=600)
        with gr.Column(scale=4):
            text1 = gr.Textbox(label="Assistant selection")
            text2 = gr.Textbox(label="Current assistant status",lines=22)
    with gr.Row():
        msg = gr.Textbox(label="Input",placeholder="What would you like to know?")
    # Some example questions
    with gr.Row():
        examples = gr.Examples(examples=[
            'What is my Alibaba Cloud account balance?',
            'Tell me about my ECS instances in the Singapore region, including their IDs, prices, and instance type details.',
            'I want to know the metrics for ecs.u1-c1m4.xlarge and ecs.gn6i-c4g1.xlarge.'],inputs=[msg])
    clear = gr.ClearButton([text1,chatbot,text2,msg])
    msg.submit(get_multi_agent_response, [msg,chatbot], [text1,chatbot,text2,msg])

if __name__ == '__main__':
    demo.launch()
