package org.example;

import org.apache.flink.api.java.utils.MultipleParameterTool;
import org.apache.flink.configuration.PipelineOptions;
import org.apache.flink.streaming.api.environment.StreamExecutionEnvironment;
import org.apache.flink.table.api.bridge.java.StreamTableEnvironment;

import org.apache.sedona.flink.SedonaContext;

public class VehicleGeofencingAnalyticsJob {

    /**
     * A Flink streaming job that performs real-time geofencing analytics.
     *
     * <p>The job consumes a stream of vehicle GPS positions, joins them with a static set of
     * predefined geographical areas (geo-fences), and then calculates the number of unique vehicles
     * within each area over a 10-second tumbling window. The results are printed to the console.
     */
    public static void main(String[] args) {
        StreamExecutionEnvironment env = StreamExecutionEnvironment.getExecutionEnvironment();
        StreamTableEnvironment tableEnv = StreamTableEnvironment.create(env);
        // register sedona functions and serializers
        SedonaContext.create(env, tableEnv);

        final MultipleParameterTool params = MultipleParameterTool.fromArgs(args);
        tableEnv.getConfig().set(PipelineOptions.GLOBAL_JOB_PARAMETERS, params.toMap());

        // create a datagen source table of vehicle positions
        tableEnv.executeSql(
                "CREATE TEMPORARY TABLE vehicle_positions\n"
                        + "(\n"
                        + "  `vehicle_id`  STRING\n"
                        + "  ,`lon`        DOUBLE\n"
                        + "  ,`lat`        DOUBLE\n"
                        + "  ,`event_time` TIMESTAMP_LTZ(3)\n"
                        + "  ,WATERMARK FOR `event_time` AS `event_time` - INTERVAL '2' SECOND\n"
                        + ")\n"
                        + "WITH (\n"
                        + "  'connector' = 'datagen'\n"
                        + "  ,'rows-per-second' = '10'\n"
                        + "  ,'fields.vehicle_id.length' = '5'\n"
                        + "  ,'fields.lon.min' = '116.3'\n"
                        + "  ,'fields.lon.max' = '116.5'\n"
                        + "  ,'fields.lat.min' = '39.85'\n"
                        + "  ,'fields.lat.max' = '40.0'\n"
                        + ")\n"
                        + ";");

        // create print sink table for displaying statistical results
        tableEnv.executeSql(
                "CREATE TEMPORARY TABLE print_sink\n"
                        + "(\n"
                        + "  `area_name`      STRING\n"
                        + "  ,`vehicle_count` BIGINT\n"
                        + "  ,`window_end`    TIMESTAMP(3)\n"
                        + ")\n"
                        + "WITH (\n"
                        + "  'connector' = 'print'\n"
                        + "  ,'print-identifier' = 'AREA_STATS_SQL'\n"
                        + ")\n"
                        + ";");

        // 1. apply 10-second tumbling window on vehicle_positions
        // 2. join with predefined geofence areas (West and East districts)
        // 3. use spatial function ST_Contains to check if vehicle is within geofence
        // 4. Group results by window
        // 5. calculate count of unique vehicles per area per window
        // 6. output results to print_sink
        tableEnv.executeSql(
                "INSERT INTO print_sink\n"
                        + "SELECT\n"
                        + "  F.area_name\n"
                        + "  ,COUNT(DISTINCT V.vehicle_id) AS vehicle_count\n"
                        + "  ,V.window_end\n"
                        + "FROM TABLE(TUMBLE(TABLE vehicle_positions, DESCRIPTOR(event_time), INTERVAL '10' SECOND)) AS V\n"
                        + "  INNER JOIN (\n"
                        + "  VALUES\n"
                        + "        ('area_01', 'West District', 'POLYGON((116.30 39.90, 116.40 39.90, 116.40 39.95, 116.30 39.95, 116.30 39.90))'),\n"
                        + "        ('area_02', 'East District', 'POLYGON((116.41 39.90, 116.48 39.90, 116.48 39.95, 116.41 39.95, 116.41 39.90))')\n"
                        + ") AS F(area_id, area_name, wkt_polygon)\n"
                        + "    ON ST_Contains(\n"
                        + "      ST_GeomFromWKT(F.wkt_polygon)\n"
                        + "      ,ST_Point(V.lon, V.lat)\n"
                        + "    )\n"
                        + "  GROUP BY\n"
                        + "    V.window_start\n"
                        + "    ,V.window_end\n"
                        + "    ,F.area_id\n"
                        + "    ,F.area_name\n"
                        + ";");
    }
}
