from flask import Flask, render_template, jsonify, request
from alibabacloud_tea_openapi.models import Config
from alibabacloud_sts20150401.client import Client as Sts20150401Client
from alibabacloud_sts20150401 import models as sts_20150401_models
import os
import json
import base64
import hmac
import datetime
import time
import hashlib

import oss2

app = Flask(__name__)

# Modify the environment variables OSS_ACCESS_KEY_ID, OSS_ACCESS_KEY_SECRET, OSS_STS_ROLE_ARN
access_key_id = os.environ.get('OSS_ACCESS_KEY_ID')
access_key_secret = os.environ.get('OSS_ACCESS_KEY_SECRET')
role_arn_for_oss_upload = os.environ.get('OSS_STS_ROLE_ARN')

# Custom session name
role_session_name = 'yourRoleSessionName'

# Replace with the actual bucket name, region_id, and host
bucket = 'examplebucket'
region_id = 'cn-hangzhou'
host = 'http://examplebucket.oss-cn-hangzhou.aliyuncs.com'

# Specify the expiration time in seconds
expire_time = 1000

# Specify the prefix for files uploaded to OSS
upload_dir = 'dir'

def hmacsha256(key, data):
    """
    Function to calculate the HMAC-SHA256 hash value
    :param key: The key used for hashing, in byte type
    :param data: The data to be hashed, in string type
    :return: The calculated HMAC-SHA256 hash value, in byte type
    """
    try:
        mac = hmac.new(key, data.encode(), hashlib.sha256)
        hmacBytes = mac.digest()
        return hmacBytes
    except Exception as e:
        raise RuntimeError(f"Failed to calculate HMAC-SHA256 due to {e}")

@app.route("/")
def hello_world():
    return render_template('index.html')

@app.route('/get_post_signature_for_oss_upload', methods=['GET'])
def generate_upload_params():
    # Initialize configuration with credentials
    config = Config(
        region_id=region_id,
        access_key_id=access_key_id,
        access_key_secret=access_key_secret
    )

    # Create STS client and obtain temporary credentials
    sts_client = Sts20150401Client(config=config)
    assume_role_request = sts_20150401_models.AssumeRoleRequest(
        role_arn=role_arn_for_oss_upload,
        role_session_name=role_session_name
    )
    response = sts_client.assume_role(assume_role_request)
    token_data = response.body.credentials.to_map()

    # Use the temporary credentials returned by STS
    temp_access_key_id = token_data['AccessKeyId']
    temp_access_key_secret = token_data['AccessKeySecret']
    security_token = token_data['SecurityToken']

    now = int(time.time())
    # Convert timestamp to datetime object
    dt_obj = datetime.datetime.utcfromtimestamp(now)
    # Add 3 hours to the current time and set it as the request expiration time
    dt_obj_plus_3h = dt_obj + datetime.timedelta(hours=3)

    # Request time
    dt_obj_1 = dt_obj.strftime('%Y%m%dT%H%M%S') + 'Z'
    # Request date
    dt_obj_2 = dt_obj.strftime('%Y%m%d')
    # Request expiration time
    expiration_time = dt_obj_plus_3h.strftime('%Y-%m-%dT%H:%M:%S.000Z')

    # Define the Base64 encoding function for callback parameters.
    def encode_callback(callback_params):
        cb_str = json.dumps(callback_params).strip()
        return oss2.compat.to_string(base64.b64encode(oss2.compat.to_bytes(cb_str)))
    
    # Build callback configuration and Base64 encode
    callback_config = {
        "callbackUrl": "http://192.168.0.0:8888/callback",  # Replace with your callback server address
        "callbackBody": "bucket=${bucket}&object=${object}&etag=${etag}&size=${size}",
        "callbackBodyType": "application/x-www-form-urlencoded"
    }
    encoded_callback = encode_callback(callback_config)

    # Build Policy and generate signature
    policy = {
        "expiration": expiration_time,
        "conditions": [
            ["eq", "$success_action_status", "200"],
            {"x-oss-signature-version": "OSS4-HMAC-SHA256"},
            {"x-oss-credential": f"{temp_access_key_id}/{dt_obj_2}/cn-hangzhou/oss/aliyun_v4_request"},
            {"x-oss-security-token": security_token},
            {"x-oss-date": dt_obj_1},
        ]
    }
    policy_str = json.dumps(policy).strip()

    # Step 2: Construct the string to sign (StringToSign)
    stringToSign = base64.b64encode(policy_str.encode()).decode()

    # Step 3: Calculate SigningKey
    dateKey = hmacsha256(("aliyun_v4" + temp_access_key_secret).encode(), dt_obj_2)
    dateRegionKey = hmacsha256(dateKey, "cn-hangzhou")
    dateRegionServiceKey = hmacsha256(dateRegionKey, "oss")
    signingKey = hmacsha256(dateRegionServiceKey, "aliyun_v4_request")

    # Step 4: Calculate Signature
    result = hmacsha256(signingKey, stringToSign)
    signature = result.hex()

    # Organize the response data
    response_data = {
        'policy': stringToSign,  # Form field
        'x_oss_signature_version': "OSS4-HMAC-SHA256",  # Specify the version and algorithm of the signature, fixed value OSS4-HMAC-SHA256
        'x_oss_credential': f"{temp_access_key_id}/{dt_obj_2}/cn-hangzhou/oss/aliyun_v4_request",  # Indicate the parameter set for deriving the key
        'x_oss_date': dt_obj_1,  # Request time
        'signature': signature,  # Signature authentication description information
        'host': host,
        'dir': upload_dir,
        'security_token': security_token,  # Security token
        'callback': encoded_callback   # Return Base64 encoded callback configuration
    }

    return jsonify(response_data)


if __name__ == "__main__":
    app.run(host="127.0.0.1", port=8000)  # If you need to listen to other addresses like 0.0.0.0, you need to add authentication mechanisms on the server side