//
//  PushService.h
//  MTPotal
//
//  Created by yangwei on 2017/10/24.
//  Copyright © 2017年 Alibaba. All rights reserved.
//

#import <Foundation/Foundation.h>

@interface PushService : NSObject

/**
 *  设备的 deviceToken
 *  注意：会在监听UIApplicationDidFinishLaunchingNotification 通知后去注册deviceToken，接入应用请务必在注册成功的回调didRegisterForRemoteNotificationsWithDeviceToken 中，初始化此值，以便上报到服务端
 *
 */
@property(nonatomic, strong) NSData *deviceToken; // 设备的 deviceToken


/**
 * 设置Push服务的url，需要在sharedService方法之前调用。
 * @param gatewayURL Push服务地址
 * @return
 */
+ (void)setPushGateway:(NSString *)gatewayURL;

+ (void)registerRemoteToken;

+ (instancetype)sharedService;

/**
 *  将设备的deviceToken 与 当前应用的 userId绑定，以支持通过 userId 推送消息
 *  注意：调用此方法，服务端会自动将此设备之前的绑定关系解绑后，再重新绑定当前输入的userid
 *
 *  @param userId 当前应用的 userId
 *
 *  @return
 */
- (void)pushBindWithUserId:(NSString *)userId completion:(void (^)(NSException *error))completion;

/**
 *  将设备的deviceToken 与 当前应用的 userId绑定，以支持通过 userId 推送消息
 *  若在绑定接口上传手机号码后则可使用短信补偿服务
 *  注意：调用此方法，服务端会自动将此设备之前的绑定关系解绑后，再重新绑定当前输入的userid
 *
 *  @param userId 当前应用的 userId
 *  @param phoneNumber 手机号码
 *
 *  @return
 */
- (void)pushBindWithUserId:(NSString *)userId phoneNumber:(NSString *)phoneNumber completion:(void (^)(NSException *error))completion;

/**
 *  解除设备的deviceToken 与 当前应用的 userId的绑定关系，如在切换账号后，需重新绑定时调用
 *
 *  @param userId 当前应用的 userId
 *
 *  @return
 */
- (void)pushUnBindWithUserId:(NSString *)userId completion:(void (^)(NSException *error))completion;

/**
 * 打开推送消息的上报接口，用于统计推送消息的打开率
 * @param  userInfo 消息的 userInfo
 * @return
 */
- (void)pushOpenLogReport:(NSDictionary *)userInfo;

typedef NS_ENUM(NSInteger, MPPushServiceLiveActivityFilterType){
    MPPushServiceLiveActivityFilterAbandon,//直接抛弃，不给回调
    MPPushServiceLiveActivityFilterCall,//过滤掉本次请求，给失败回调（callRepeat）
    MPPushServiceLiveActivityFilterRefuse//不做过滤
};

/**
 * 上报实时活动的token，并与activityId绑定
 * @param activityId 实时活动的id，用户自定义
 * @param pushToken 实时活动获取到的token
 * @param filter 由于iPhone 的pushTokenUpdates，会同时被调用两次；在多个LiveActivity的场景，新创建LiveActivity时之前的LiveActivity pushTokenUpdates又会被重新唤醒
 * 所以sdk提供了过滤功能
 *  filter 为MPPushServiceLiveActivityFilterAbandon，sdk会自动直接抛弃重复的调用，不给回调
 *
 *  filter 为MPPushServiceLiveActivityFilterCall ，sdk会自动过滤掉本次请求，给失败回调（callRepeat）
 *  此时 error.reason为@"callRepeat",请忽略
 *
 *  filter 为MPPushServiceLiveActivityFilterReRefuse，sdk内部不做过滤
 *  【重复的调用】相同的activityId，相同的pushToken；如果是上报失败了，客户端重新上报不会被认为是相同的调用
 *
 * @param completion completion，error.reason为@"callRepeat"时为 重复的调用，请忽略
 *
 * @return
 */
- (void)liveActivityBindWithActivityId:(NSString *)activityId pushToken:(NSString *)pushToken filter:(MPPushServiceLiveActivityFilterType)filter completion:(void (^)(NSException *error))completion;

@end
