/**
 * \file sha256.h
 *
 * \brief SHA-224 and SHA-256 cryptographic hash function
 *
 *  Copyright (C) 2006-2015, ARM Limited, All Rights Reserved
 *  SPDX-License-Identifier: Apache-2.0
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"); you may
 *  not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 *  WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 *
 *  This file is part of mbed TLS (https://tls.mbed.org)
 */
#ifndef ANTSSM_SHA256_H
#define ANTSSM_SHA256_H

#if !defined(ANTSSM_CONFIG_FILE)
#include "config.h"
#else
#include ANTSSM_CONFIG_FILE
#endif

#include <stddef.h>
#include <stdint.h>

#include "log.h"

#if !defined(ANTSSM_SHA256_ALT)
// Regular implementation
//

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \brief          SHA-256 context structure
 */
typedef struct {
    uint32_t total[2];          /*!< number of bytes processed  */
    uint32_t state[8];          /*!< intermediate digest state  */
    unsigned char buffer[64];   /*!< data block being processed */
    int is224;                  /*!< 0 => SHA-256, else SHA-224 */
    char traceid[ANTSSM_TRACEID_LENGTH];
} mpaas_antssm_sha256_context_t;

/**
 * \brief          Initialize SHA-256 context
 *
 * \param ctx      SHA-256 context to be initialized
 */
void mpaas_antssm_sha256_init(mpaas_antssm_sha256_context_t *ctx);

/**
 * \brief          Clear SHA-256 context
 *
 * \param ctx      SHA-256 context to be cleared
 */
void mpaas_antssm_sha256_free(mpaas_antssm_sha256_context_t *ctx);

/**
 * \brief          Clone (the state of) a SHA-256 context
 *
 * \param dst      The destination context
 * \param src      The context to be cloned
 */
void mpaas_antssm_sha256_clone(mpaas_antssm_sha256_context_t *dst,
                         const mpaas_antssm_sha256_context_t *src);

/**
 * \brief          SHA-256 context setup
 *
 * \param ctx      context to be initialized
 * \param is224    0 = use SHA256, 1 = use SHA224
 */
void mpaas_antssm_sha256_starts(mpaas_antssm_sha256_context_t *ctx, int is224);

/**
 * \brief          SHA-256 process buffer
 *
 * \param ctx      SHA-256 context
 * \param input    buffer holding the  data
 * \param ilen     length of the input data
 */
void mpaas_antssm_sha256_update(mpaas_antssm_sha256_context_t *ctx, const unsigned char *input,
                          size_t ilen);

/**
 * \brief          SHA-256 final digest
 *
 * \param ctx      SHA-256 context
 * \param output   SHA-224/256 checksum result
 */
void mpaas_antssm_sha256_finish(mpaas_antssm_sha256_context_t *ctx, unsigned char output[32]);

/* Internal use */
void mpaas_antssm_sha256_process(mpaas_antssm_sha256_context_t *ctx, const unsigned char data[64]);

#ifdef __cplusplus
}
#endif

#else  /* ANTSSM_SHA256_ALT */
#include "sha256_alt.h"
#endif /* ANTSSM_SHA256_ALT */

#ifdef __cplusplus
extern "C" {
#endif

/**
 * \brief          Output = SHA-256( input buffer )
 *
 * \param input    buffer holding the  data
 * \param ilen     length of the input data
 * \param output   SHA-224/256 checksum result
 * \param is224    0 = use SHA256, 1 = use SHA224
 */
void mpaas_antssm_sha256(const unsigned char *input, size_t ilen,
                   unsigned char output[32], int is224);

/**
 * \brief          Checkup routine
 *
 * \return         0 if successful, or 1 if the test failed
 */
int mpaas_antssm_sha256_self_test(int verbose);

#ifdef __cplusplus
}
#endif

#endif /* mpaas_antssm_sha256.h */
