//
//  FLTracker.h
//  FullLinkTracker
//
//  Created by llb on 2018/10/16.
//  Copyright © 2018年 Alipay. All rights reserved.
//

#import <Foundation/Foundation.h>

typedef NS_OPTIONS(NSUInteger, FLExceptionLevel)
{
    kFLExceptionLevelNone = 0,
    kFLExceptionLevelInfo,      // 信息级别，不影响业务正常使用，不会立即触发日志上报
    kFLExceptionLevelWarn,      // 警告级别，不影响业务正常使用，不会立即触发日志上报
    kFLExceptionLevelError,     // 错误级别，影响业务正常使用，会立即触发日志上报
};

NS_ASSUME_NONNULL_BEGIN

@class FLPage, FLException;

@interface FLTracker : NSObject

#pragma mark - 性能桩点--时间戳埋点接口

/**
 * 记录时间桩点。注意：底层使用的机器时间，单位是毫秒。CACurrentMediaTime() * 1000
 * @param stubId     桩点位关键字
 * @param clusterId  点位集id
 * @param bizType    业务id
 */
+ (void)logStub:(NSString *)stubId
      clusterId:(NSString *)clusterId
        bizType:(NSString *)bizType;

/**
 * 记录时间桩点。
 * 注意：底层使用的机器时间，单位是毫秒。CACurrentMediaTime() * 1000
 *
 * @param stubId     桩点位关键字
 * @param time       业务指定时间戳，为nil时，接口里取当前时间
 * @param clusterId  点位集id
 * @param bizType    业务id
 */
+ (void)logStub:(NSString *)stubId
           time:(NSTimeInterval)time
      clusterId:(NSString *)clusterId
        bizType:(NSString *)bizType;

/**
 * 记录时间桩点。注意：底层使用的机器时间，单位是毫秒。CACurrentMediaTime() * 1000
 * @param stubId        桩点位关键字
 * @param clusterId     点位集id
 * @param bizType       业务id
 * @param canOverride   是否覆盖原有值
 */
+ (void)logStub:(NSString *)stubId
      clusterId:(NSString *)clusterId
        bizType:(NSString *)bizType
    canOverride:(BOOL)canOverride;

/**
 * 记录时间桩点。
 * 注意：底层使用的机器时间，单位是毫秒。CACurrentMediaTime() * 1000
 *
 * @param stubId        桩点位关键字
 * @param time          业务指定时间戳，为nil时，接口里取当前时间
 * @param clusterId     点位集id
 * @param bizType       业务id
 * @param canOverride   是否覆盖原有值
 */
+ (void)logStub:(NSString *)stubId
           time:(NSTimeInterval)time
      clusterId:(NSString *)clusterId
        bizType:(NSString *)bizType
    canOverride:(BOOL)canOverride;


#pragma mark - 性能桩点--耗时埋点接口

/**
 * 记录处理耗时开始。
 * 注意：底层使用的机器时间，单位是毫秒。CACurrentMediaTime() * 1000
 *
 * @param costId    耗时关键字
 * @param clusterId 点位集id
 * @param bizType   业务id
 */
+ (void)logCostStart:(NSString *)costId
           clusterId:(NSString *)clusterId
             bizType:(NSString *)bizType;

/**
 * 记录处理耗时结束。
 * 注意：底层使用的机器时间，单位是毫秒。CACurrentMediaTime() * 1000
 *
 * @param costId    耗时关键字
 * @param clusterId 点位集id
 * @param bizType   业务id
 */
+ (void)logCostEnd:(NSString *)costId
         clusterId:(NSString *)clusterId
           bizType:(NSString *)bizType;

/**
 * 记录处理耗时结束。
 * 注意：底层使用的机器时间，单位是毫秒。CACurrentMediaTime() * 1000
 *
 * @param costId    耗时关键字
 * @param costTime  指定耗时时长
 * @param clusterId 点位集id
 * @param bizType   业务id
 */
+ (void)logCost:(NSString *)costId
       costTime:(NSTimeInterval)costTime
      clusterId:(NSString *)clusterId
        bizType:(NSString *)bizType;

/**
 * 记录处理耗时结束。
 * 注意：底层使用的机器时间，单位是毫秒。CACurrentMediaTime() * 1000
 *
 * @param costId        耗时关键字
 * @param costTime      指定耗时时长
 * @param clusterId     点位集id
 * @param bizType       业务id
 * @param canOverride   是否覆盖原有值
 */
+ (void)logCost:(NSString *)costId
       costTime:(NSTimeInterval)costTime
      clusterId:(NSString *)clusterId
        bizType:(NSString *)bizType
    canOverride:(BOOL)canOverride;


#pragma mark - 维度桩点--维度条件埋点接口

/**
 * 记录维度信息。以字典格式上报，因此key唯一。
 *
 * @param key       关键字
 * @param value     信息
 * @param clusterId 点位集id
 * @param bizType   业务id
 */
+ (void)logEnvInfo:(NSString *)key
             value:(NSString *)value
         clusterId:(NSString *)clusterId
           bizType:(NSString *)bizType;

/**
 * 记录业务信息。以字典格式上报，因此key唯一。
 *
 * @param key           关键字
 * @param value         信息
 * @param clusterId     点位集id
 * @param bizType       业务id
 * @param canOverride   是否覆盖原有值
 */
+ (void)logEnvInfo:(NSString *)key
             value:(NSString *)value
         clusterId:(NSString *)clusterId
           bizType:(NSString *)bizType
       canOverride:(BOOL)canOverride;


#pragma mark - 异常桩点--异常埋点接口

/**
 * 记录异常，默认异常不会影响业务处理，待定页面显示后再上报
 *
 * @param exceptionId  服务端申请异常关键字
 * @param reason       异常描述
 * @param clusterId    点位集id
 * @param bizType      业务id
 */
+ (void)logException:(NSString *)exceptionId
              reason:(NSString *)reason
           clusterId:(NSString *)clusterId
             bizType:(NSString *)bizType;

/**
 * 记录异常。
 * @param exceptionId  服务端申请异常关键字
 * @param reason       异常描述
 * @param clusterId    点位集id
 * @param bizType      业务id
 * @param level        异常级别，注意：kFLExceptionLevelError触发日志立即上报
 */
+ (void)logException:(NSString *)exceptionId
              reason:(NSString *)reason
           clusterId:(NSString *)clusterId
             bizType:(NSString *)bizType
               level:(FLExceptionLevel)level;

/**
 * 上报异常
 *
 * @param exception  异常信息
 */
+ (void)logException:(FLException *)exception;


#pragma mark - 业务自定义桩点--业务信息埋点接口

/**
 * 记录业务信息。以字典格式上报，因此key唯一。
 * 注意：该信息只在明细日志里上报，链路日志里不上报。
 *
 * @param key       关键字
 * @param value     信息
 * @param clusterId 点位集id
 */
+ (void)logBizInfo:(NSString *)key
             value:(NSString *)value
         clusterId:(NSString *)clusterId;

/**
 * 记录业务信息。以字典格式上报，因此key唯一。
 *
 * @param key           关键字
 * @param value         信息
 * @param clusterId     点位集id
 * @param canOverride   是否覆盖原有值
 */
+ (void)logBizInfo:(NSString *)key
             value:(NSString *)value
         clusterId:(NSString *)clusterId
       canOverride:(BOOL)canOverride;

/**
 * 记录业务信息。以数组格式上报，info里key可以重复。
 * 注意：该信息只在明细日志里上报，链路日志里不上报。
 *
 * @param info      信息，作为数组元素
 * @param clusterId 点位集id
 */
+ (void)logBizInfo:(NSDictionary *)info
         clusterId:(NSString *)clusterId;


#pragma mark - 诊断桩点--诊断埋点接口

/**
 * 记录链路诊断信息。以字典格式上报，因此key唯一。
 *
 * @param key       关键字
 * @param value     信息
 * @param clusterId 点位集id
 * @param bizType   业务id
 * @param linkName  所属链路名字
 */
+ (void)logDiagnose:(NSString *)key
              value:(NSString *)value
          clusterId:(NSString *)clusterId
            bizType:(NSString *)bizType
           linkName:(NSString *)linkName;

/**
 * 记录明细诊断信息。以字典格式上报，因此key唯一。
 *
 * @param key           关键字
 * @param value         信息
 * @param clusterId     点位集id
 * @param bizType       业务id
 * @param canOverride   是否覆盖原有值
 */
+ (void)logDiagnose:(NSString *)key
              value:(NSString *)value
          clusterId:(NSString *)clusterId
            bizType:(NSString *)bizType
        canOverride:(BOOL)canOverride;

#pragma mark - 设置页面相关信息接口

/**
 * 设置页面相关信息。
 *
 * @param pageInfo  页面信息
 * @param clusterId 点位集id
 */
+ (void)setPageInfo:(FLPage *)pageInfo
          clusterId:(NSString *)clusterId;

#pragma mark - 自定义点位集接口
/**
 * 提交自定义点位集内所有点位信息。注意：单独上报，不参加全链路匹配上报。
 *
 * @param clusterId 点位集id
 * @param bizType   业务id
 */
+ (void)commitCluster:(NSString *)clusterId bizType:(NSString *)bizType;

/**
 * 提交自定义点位集内所有点位信息，同时参加全链路匹配。
 *
 * @param clusterId 点位集id
 * @param bizType   业务id
 */
+ (void)commitClusterAndFullLink:(NSString *)clusterId bizType:(NSString *)bizType;

/**
 * 取消自定义点位集内所有点位信息，不上报。
 *
 * @param clusterId 点位集id
 * @param bizType   业务id
 */
+ (void)cancelCluster:(NSString *)clusterId bizType:(NSString *)bizType;

/**
 * 在指定点位集ID后面加“__+flt+时间戳”生成唯一标识，用于避免自定义埋点并发埋点问题：
 * 如果业务自定义埋点存在并发情况时，因为clusterId是固定字符串，所以导致并发埋点信息混合在一起。
 *
 * @param origin 指定id
 *
 * @return 成功返回origin__时间戳+flt+，否则返回nil
 */
+ (NSString *)generateFltId:(NSString *)origin;

/*
* @return 返回去掉__时间戳+flt+后缀真实的点位集ID
*/
+ (NSString *)return2Origin:(NSString *)fltId;

#pragma mark - ABTest信息关联接口
/**
 * 上报APTest实验信息。注意：监控指标和ABText相关才需要配置，否则不要配置。
 * 上报配置服务进行APTest信息，从而实现和实验相关指标自动有实验维度，无需修改监控大盘指标。
 *
 * @param testList  AB实验关键字列表
 * @param clusterId 点位集id
 */
+ (void)logABTestInfo:(NSArray *)testList clusterId:(NSString *)clusterId;


#pragma mark - 设置sessionId相关接口
/**
 * sessionId接口，用于将包含sessionId的明细合并为一个链路日志。
 * 保存在埋点env字段，对应key：flt_sessionId。
 *
 * @param sessionId   指定sessionId
 * @param clusterId   点位集id
 * @param isFirstNode 是否是整个sessionId关联链路第一个节点
 */
+ (void)logSessionId:(NSString *)sessionId
           clusterId:(NSString *)clusterId
         isFirstNode:(BOOL)isFirstNode;

/**
 * sessionId接口，用于将包含sessionId的明细合并为一个链路日志。
 * 保存在埋点env字段，对应key：flt_sessionId。
 *
 * @param sessionId   指定sessionId
 * @param clusterId   点位集id
 * @param targetAppId 目标app id
 * @param isFirstNode 是否是整个sessionId关联链路第一个节点
 */
+ (void)logSessionId:(NSString *)sessionId
         targetAppId:(nullable NSString *)appId
           clusterId:(NSString *)clusterId
         isFirstNode:(BOOL)isFirstNode;

@end

NS_ASSUME_NONNULL_END
