//
//  SPMTrackerLog.h
//  SPMTracker
//
//  Created by BoTao on 2017/2/3.
//  Copyright © 2017年 Alipay. All rights reserved.
//
#import <Foundation/Foundation.h>
#import <APRemoteLogging/APMonitorPointDataDefines.h>

#ifndef SPMTrackerLog_h
#define SPMTrackerLog_h

#define kSPMActionOpenPage @"openPage"
#define kSPMActionClicked @"clicked"
#define kSPMActionSlided @"slided"
#define kSPMActionExposure @"exposure"

#define kSPMMonitorChInfoKey @"chInfo"
#define kSPMPageIDKey @"pagets"
#define kSPMPageBackKey @"pageBack"

NS_ASSUME_NONNULL_BEGIN

void setDisableSPMLog(BOOL disable);

extern NSString *const SPMTRACKER_LACK_OF_PAGEID_KEY;

@interface SPMTrackerLog : NSObject

#pragma mark - 点击

// 行为埋点
/**
 *  点击-行为埋点使用
 *
 *  @param spmId SPM埋点ID
 *  @param entityId 对应scmId、rpcId
 *  @param bizCode 业务bizType，埋点平台申请
 *  @param param4 扩展参数，业务自己根据需要填充。字典会被转换成key-value的字符串计入日志
 *  @param view 埋点对应view，传vc对象最好，其次view对象，不能为空
 *  @param loggerLevel 日志等级(LoggerLevelLow/LoggerLevelMedium/LoggerLevelHigh)，默认为LoggerLevelMedium
 */
+ (void)clickedLogWithSpmId:(NSString *)spmId entityId:(NSString *)entityId bizCode:(NSString *)bizCode extParams4:(nullable NSDictionary *)param4 view:(NSObject *)view logLevel:(NSString *)logLevel ;

// 可以明确传递pageId时调用次方法，否则调用楼上的方法
+ (void)clickedLogWithSpmId:(NSString *)spmId entityId:(NSString *)entityId bizCode:(NSString *)bizCode extParams4:(nullable NSDictionary *)param4 pageId:(NSString *)pageId logLevel:(NSString *)logLevel ;

+ (void)clickedLogWithSpmId:(NSString *)spmId bizCode:(NSString *)bizCode extParams4:(nullable NSDictionary *)param4 view:(NSObject *)view logLevel:(NSString *)logLevel ;

+ (void)clickedLogWithSpmId:(NSString *)spmId bizCode:(NSString *)bizCode extParams4:(nullable NSDictionary *)param4 view:(NSObject *)view ;

#pragma mark - 内容点击

/**
 *  内容点击埋点
 *
 *  @param spmId SPM埋点ID
 *  @param bizCode 业务bizType，埋点平台申请
 *  @param scm SCM埋点ID
 *  @param newChinfo 流量来源
 *  @param param4 扩展参数，业务自己根据需要填充。字典会被转换成key-value的字符串计入日志
 *  @param view 埋点对应view，传vc对象最好，其次view对象，不能为空
 */
+ (void)clickedContentLogWithSpmId:(NSString *)spmId
                           bizCode:(NSString *)bizCode
                               scm:(NSString *)scm
                         newChinfo:(NSString *)newChinfo
                        extParams4:(nullable NSDictionary *)param4
                              view:(NSObject *)view;

/**
 *  内容点击埋点
 *
 *  @param spmId SPM埋点ID
 *  @param bizCode 业务bizType，埋点平台申请
 *  @param scm SCM埋点ID
 *  @param newChinfo 流量来源
 *  @param param4 扩展参数，业务自己根据需要填充。字典会被转换成key-value的字符串计入日志
 *  @param view 埋点对应view，传vc对象最好，其次view对象，不能为空
 *  @param loggerLevel 日志等级(LoggerLevelLow/LoggerLevelMedium/LoggerLevelHigh)，默认为LoggerLevelMedium
 */

+ (void)clickedContentLogWithSpmId:(NSString *)spmId
                           bizCode:(NSString *)bizCode
                               scm:(NSString *)scm
                         newChinfo:(NSString *)newChinfo
                        extParams4:(nullable NSDictionary *)param4
                              view:(NSObject *)view
                          logLevel:(NSString *)logLevel;

/**
 *  内容点击埋点,开启全引导，渠道链，渠道
 *
 *  @param spmId SPM埋点ID
 *  @param bizCode 业务bizType，埋点平台申请
 *  @param scm SCM埋点ID
 *  @param newChinfo 流量来源
 *  @param param4 扩展参数，业务自己根据需要填充。字典会被转换成key-value的字符串计入日志
 *  @param view 埋点对应view，传vc对象最好，其次view对象，不能为空
 *  @param loggerLevel 日志等级(LoggerLevelLow/LoggerLevelMedium/LoggerLevelHigh)，默认为LoggerLevelMedium
 */

+ (void)clickedContentTraceWithSpmId:(NSString *)spmId
                             bizCode:(NSString *)bizCode
                                 scm:(NSString *)scm
                           newChinfo:(NSString *)newChinfo
                          extParams4:(nullable NSDictionary *)param4
                                view:(NSObject *)view
                            logLevel:(NSString *)logLevel;
/**
 *  内容点击埋点,开启全引导，渠道链，渠道
 *
 *  @param spmId SPM埋点ID
 *  @param pageId （没有pageId可以传空值@"" 或 nil）
 *  @param entityId （没有entityId可以传空值@"" 或 nil）
 *  @param bizCode 业务bizType，埋点平台申请
 *  @param scm SCM埋点ID
 *  @param newChinfo 流量来源
 *  @param param4 扩展参数，业务自己根据需要填充。字典会被转换成key-value的字符串计入日志
 *  @param view 埋点对应view，传vc对象最好，其次view对象，不能为空
 *  @param loggerLevel 日志等级(LoggerLevelLow/LoggerLevelMedium/LoggerLevelHigh)，默认为LoggerLevelMedium
 */

+ (void)clickedContentTraceWithSpmId:(NSString *)spmId
                              pageId:(nullable NSString *)pageId
                            entityId:(nullable NSString *)entityId
                             bizCode:(NSString *)bizCode
                                 scm:(NSString *)scm
                           newChinfo:(NSString *)newChinfo
                          extParams4:(NSDictionary *)param4
                                view:(NSObject *)view
                            logLevel:(NSString *)logLevel;

#pragma mark - 翻页

/**
 *  翻页-行为埋点使用
 *
 *  @param spmId SPM埋点ID
 *  @param entityId 对应scmId、rpcId
 *  @param bizCode 业务bizType，埋点平台申请
 *  @param param4 扩展参数，业务自己根据需要填充。字典会被转换成key-value的字符串计入日志
 *  @param view 埋点对应view，传vc对象最好，其次view对象，不能为空
 *  @param loggerLevel 日志等级(LoggerLevelLow/LoggerLevelMedium/LoggerLevelHigh)，默认为LoggerLevelMedium
 */
+ (void)slideLogWithSpmId:(NSString *)spmId entityId:(NSString *)entityId bizCode:(NSString *)bizCode extParams4:(nullable NSDictionary *)param4 view:(NSObject *)view logLevel:(NSString *)logLevel ;

+ (void)slideLogWithSpmId:(NSString *)spmId bizCode:(NSString *)bizCode extParams4:(nullable NSDictionary *)param4 view:(NSObject *)view logLevel:(NSString *)logLevel ;

+ (void)slideLogWithSpmId:(NSString *)spmId bizCode:(NSString *)bizCode extParams4:(nullable NSDictionary *)param4 view:(NSObject *)view ;

#pragma mark - 曝光

/**
 *  曝光-行为埋点使用
 *
 *  @param spmId SPM埋点ID
 *  @param entityId 对应scmId、rpcId
 *  @param bizCode 业务bizType，埋点平台申请
 *  @param param4 扩展参数，业务自己根据需要填充。字典会被转换成key-value的字符串计入日志
 *  @param view 埋点对应view，传vc对象最好，其次view对象，不能为空
 *  @param loggerLevel 日志等级(LoggerLevelLow/LoggerLevelMedium/LoggerLevelHigh)，默认为LoggerLevelMedium
 */
+ (void)exposureLogWithSpmId:(NSString *)spmId entityId:(NSString *)entityId bizCode:(NSString *)bizCode extParams4:(nullable NSDictionary *)param4 view:(NSObject *)view logLevel:(NSString *)logLevel ;

// 可以明确传递pageId时调用次方法，否则调用楼上的方法
+ (void)exposureLogWithSpmId:(NSString *)spmId entityId:(NSString *)entityId bizCode:(NSString *)bizCode extParams4:(nullable NSDictionary *)param4 pageId:(NSString *)pageId logLevel:(NSString *)logLevel ;

+ (void)exposureLogWithSpmId:(NSString *)spmId bizCode:(NSString *)bizCode extParams4:(nullable NSDictionary *)param4 view:(NSObject *)view logLevel:(NSString *)logLevel ;

+ (void)exposureLogWithSpmId:(NSString *)spmId bizCode:(NSString *)bizCode extParams4:(nullable NSDictionary *)param4 view:(NSObject *)view ;

#pragma mark - 内容曝光

/**
 *  内容曝光埋点
 *
 *  @param spmId SPM埋点ID
 *  @param bizCode 业务bizType，埋点平台申请
 *  @param scm SCM埋点ID
 *  @param newChinfo 流量来源
 *  @param param4 扩展参数，业务自己根据需要填充。字典会被转换成key-value的字符串计入日志
 *  @param view 埋点对应view，传vc对象最好，其次view对象，不能为空
 */
+ (void)exposureContentLogWithSpmId:(NSString *)spmId
                            bizCode:(NSString *)bizCode
                                scm:(NSString *)scm
                          newChinfo:(NSString *)newChinfo
                         extParams4:(nullable NSDictionary *)param4
                               view:(NSObject *)view;

/**
 *  内容曝光埋点
 *
 *  @param spmId SPM埋点ID
 *  @param bizCode 业务bizType，埋点平台申请
 *  @param scm SCM埋点ID
 *  @param newChinfo 流量来源
 *  @param param4 扩展参数，业务自己根据需要填充。字典会被转换成key-value的字符串计入日志
 *  @param view 埋点对应view，传vc对象最好，其次view对象，不能为空
 *  @param loggerLevel 日志等级(LoggerLevelLow/LoggerLevelMedium/LoggerLevelHigh)，默认为LoggerLevelMedium
 */
+ (void)exposureContentLogWithSpmId:(NSString *)spmId
                            bizCode:(NSString *)bizCode
                                scm:(NSString *)scm
                          newChinfo:(NSString *)newChinfo
                         extParams4:(nullable NSDictionary *)param4
                               view:(NSObject *)view
                           logLevel:(NSString *)logLevel;

#pragma mark - 合并曝光
/*
 *  合并曝光
 *  推荐使用，解决列表元素曝光需求，每个元素不会重复曝光，储存曝光日志，达到上报条件时，多条曝光日志合并后上报埋点
 *  合并曝光文档：https://lark.alipay.com/data/help/willav
 *
 *  @param spmId: 埋点SPMID，不可为空
 *  @param entityId 对应scmId、rpcId
 *  @param bizCode 业务bizType，埋点平台申请
 *  @param rid: RPC唯一标示，可以时tracerid、recommondid等
 *  @param pos: cell索引号，类似SPMID的N
 *  @param extParams: 扩展参数，业务自己根据需要填充。字典会被转换成key-value的字符串计入日志
 *  @param view: 埋点对应view，传vc对象最好，其次view对象，不能为空
 *  @param loggerLevel 日志等级(LoggerLevelLow/LoggerLevelMedium/LoggerLevelHigh)，默认为LoggerLevelMedium
 */
+ (void)mergerExposureLogWithSpmId:(NSString *)spmId entityId:(NSString *)entityId rid:(NSString *)rid position:(NSString *)pos bizCode:(NSString *)bizCode extParams:(NSDictionary *)extParams view:(NSObject *)view logLevel:(NSString *)logLevel;

// 可以明确传递pageId时调用次方法，否则调用楼上的方法
+ (void)mergerExposureLogWithSpmId:(NSString *)spmId entityId:(NSString *)entityId rid:(NSString *)rid position:(NSString *)pos bizCode:(NSString *)bizCode extParams:(NSDictionary *)extParams pageId:(NSString *)pageId logLevel:(NSString *)logLevel;

+ (void)mergerExposureLogWithSpmId:(NSString *)spmId rid:(NSString *)rid position:(NSString *)pos bizCode:(NSString *)bizCode extParams:(NSDictionary *)extParams view:(NSObject *)view logLevel:(NSString *)logLevel;

+ (void)mergerExposureLogWithSpmId:(NSString *)spmId rid:(NSString *)rid position:(NSString *)pos bizCode:(NSString *)bizCode extParams:(NSDictionary *)extParams view:(NSObject *)view;

/*
 *  上报合并曝光日志，上报成功后会清除已上报日志
 *  pageMonitor的onViewWillDisappear中会调用，如果业务需要手工触发上报合并曝光日志，调用此方法
 *
 *  @param spmId: 埋点SPMID，所有向后匹配的spmid都上报，例如a13.b42，则a13.b42.c11、a13.b42.c51.d222等合并曝光日志都上报并清除记录
 *  @param view: 埋点对应view，传vc对象最好，其次view对象，不能为空
 */
+ (void)commiteMergerExposureLogWithSpmId:(NSString *)spmId view:(NSObject *)view;

#pragma mark - 日志工具方法
/**
 *  @param actionId  action ID 必填，标识一个埋点的类别，可用actionID定义在APMonitorPointDataDefines.h文件中
 *  @param extParams 扩展参数，业务自己根据需要填充。数组元素最多4个，第四个元素可以是字典，字典内容可自定义，字典会被转换成key-value的字符串计入日志
 *  @param spmId      埋点ID（后端定义的，找@望乡生成）
 *  @param entityId 对应scmId、rpcId
 *  @param bizCode 业务bizType，埋点平台申请
 *  @param view 埋点对应view，传vc对象最好，其次view对象，不能为空
 *  @param loggerLevel 日志等级(LoggerLevelLow/LoggerLevelMedium/LoggerLevelHigh)，默认为LoggerLevelMedium
 */
+ (void)writeLogWithActionId:(NSString *)actionId extParams:(NSArray *)extParams spmId:(NSString *)spmId entityId:(NSString *)entityId bizCode:(NSString *)bizCode view:(NSObject *)view logLevel:(NSString *)logLevel;

+ (void)writeLogWithActionId:(NSString *)actionId extParams:(NSArray *)extParams spmId:(NSString *)spmId bizCode:(NSString *)bizCode view:(nonnull NSObject *)view logLevel:(NSString *)logLevel;

/**
 *  pageMonitor，记录chinfo信息，一般在scheme跳转的方法中最后调用该方法，logPageEndWithSpmId时使用chinfo信息，用完删除对应记录
 *
 *  @param options scheme参数
 *  @param viewController当前vc，作为记录信息的key
 *
 *  @return 参数包含chinfo时，返回YES，否则NO
 */
+ (BOOL)spliteSchemaOptions:(NSDictionary *)options viewController:(NSObject *)viewController;

/**
 *  pageMonitor，更新chinfo信息
 *
 *  @param info chInfo值
 *  @param viewController当前vc，作为记录信息的key
 *
 */
+ (void)updateChInfo:(NSString *)info viewController:(NSObject *)viewController;

/**
 *  pageMonitor，根据vc获取chInfo
 */
+ (NSString *)chInfoForViewController:(NSObject *)vc;

#pragma mark - 页面埋点
/**
 *  pageMonitor，页面打开
 *
 *  @param spmId 代表当前页面的spm
 *  @param view 埋点对应view，传vc对象最好，其次view对象，不能为空
 *
 */
+ (void)onViewDidAppear:(NSString *)spmId view:(nonnull NSObject *)view DEPRECATED_MSG_ATTRIBUTE("手动埋点已废弃，请使用UEP");
+ (void)onViewDidAppear:(NSString *)spmId view:(nonnull NSObject *)view scm:(NSString *)scm;

/**
 *  pageMonitor，页面结束
 *  @param spmId 代表当前页面的spm
 *  @param view 埋点对应view，传vc对象最好，其次view对象，不能为空
 *  @param bizCode 业务bizCode
 *  @param param 手动埋点扩展参数4，底层保留key不要使用：chInfo
 */
+ (void)onViewWillDisappear:(NSString *)spmId view:(nonnull NSObject *)view bizCode:(NSString *)bizCode param:(NSDictionary *)param DEPRECATED_MSG_ATTRIBUTE("手动埋点已废弃，请使用UEP");

/**
 *  清空pageid对应的所有埋点缓存数据，非必需调用
 *  @param view 埋点对应view，传vc对象最好，其次view对象，不能为空
 */

+ (void)onDealloc:(nonnull NSObject *)view DEPRECATED_MSG_ATTRIBUTE("手动埋点已废弃，请使用UEP");

/**
*  新增容器页面打开方法，H5专用
*/
+ (void)onNebulaPageStartWithSpmId:(NSString *)spmId index:(NSObject *)index type:(SPMPageType) type;
+ (void)onNebulaPageStartWithSpmId:(NSString *)spmId index:(NSObject *)index type:(SPMPageType) type scm:(NSString *)scm;
/**
 * 对应onViewDidAppear:view:，H5专用
 */
+ (void)logPageStartWithSpmId:(NSString *)spmId index:(NSObject *)index;

/**
 * 对应onViewWillDisappear:view:bizCode:param:，H5专用
 */
+ (void)logPageEndWithSpmId:(NSString *)spmId
                      index:(NSObject *)index
                    bizType:(nullable NSString *)bizType
                      param:(nullable NSDictionary *)param;



///////////////////////////////////////////////////////////////////////////////////////////////////////////
// pagets
/**
 *  RPC透传pageId，根据view获得pageId
 *
 *  @param view 埋点对应view，传vc对象最好，其次view对象，不能为空
 *
 *  @return 以pageId为value的字典
 **/
+ (NSDictionary* )headerParamsForPagets:(NSObject *)view;

/**
 *  根据view获得miniPageId
 *
 *  @param view 埋点对应view，传vc对象最好，其次view对象，不能为空
 *
 *  @return miniPageId值
 **/
+ (NSString *)getMiniPageId:(NSObject *)view;

/**
 *  根据view获得pageId
 *
 *  @param view 埋点对应view，传vc对象最好，其次view对象，不能为空
 *
 *  @return pageId值
 **/
+ (NSString *)getPageId:(NSObject *)view;

///////////////////////////////////////////////////////////////////////////////////////////////////////////
// pageback
/**
 *  设备页面回退标志黑名单
 *
 *  @param pageBackDicBlack 不会被标记为退回的页面们的spmid，比如：钱包首页4大金刚之一口碑首页a13.b42
 **/
+(void)updatePageBackDicBlack:(NSArray<NSString *> *)pageBackDicBlack;

#pragma mark - 全引导页面参数

/**
 *  设置页面参数
 *  @param params 页面参数 {$chinfo, $scm}
 *  @param view 页面对象，必须与页面埋点时使用的view一致
 */
+ (void)setPageParams:(NSDictionary *)params
              forView:(NSObject *)view;

/**
 *  设置页面参数
 *  @param params 页面参数 {$chinfo, $scm}
 *  @param view 页面
 *  @param step 透传步数，支持【3-5】，最少透传到p-pre2，最多透传到p-pre4
 */
+ (void)setPageParams:(NSDictionary *)params
              forView:(NSObject *)view
            traceStep:(NSUInteger)step;

/**
 *  设置页面参数
 *  @param params 页面参数 {$chinfo, $scm}
 *  @param view 页面
 *  @param step 透传步数，支持【3-5】，最少透传到p-pre2，最多透传到p-pre4
 *  @param isForce 是否强制更新当前
 */
+ (void)setPageParams:(NSDictionary *)params
              forView:(NSObject *)view
            traceStep:(NSUInteger)step
              isForce:(BOOL)isForce;

/**
 *  设置下一个页面参数
 *  @param params 页面参数 {$chinfo, $scm}
 */
+ (void)setNextPageParams:(NSDictionary *)params;

/**
 *  设置下一个页面参数
 *  @param params 页面参数 {$chinfo, $scm}
 *  @param step 透传步数，支持【3-5】，最少透传到p-pre2，最多透传到p-pre4
 */
+ (void)setNextPageParams:(NSDictionary *)params
                traceStep:(NSUInteger)step;

/**
 *  获取当前页面全引导参数
 *  @retrn params 页面参数  {"p-root" : {$chinfo, $scm}, "p-pre": {$chinfo, $scm} , ...}
 */
+ (nullable NSDictionary *)getCurrentPageParams;

#pragma mark - 渠道链
/**
 * 点击时添加渠道
 * @param chinfo 渠道 需要传入JSON字典格式字符串！
 *
 */
+ (void)appendChinfoWhenClick:(NSString *)chinfo;

/**
 * 页面打开时添加渠道
 * @param chinfo 渠道 需要传入JSON字典格式字符串！
 *
 */
+ (void)appendChinfoWhenPageOpen:(NSString *)chinfo;

/**
 *  获取渠道链
 *  @return 渠道链
 *
 */
+ (NSString *)chinfoChain;

/**
 *  返回回退识别过滤的渠道链
 *  @return 渠道链
 *
 */
+ (NSString *)backFilteredChinfoChain;

/**
 *  更新渠道链，只支持更新渠道链最后一个元素，需要匹配相同的chinfo
 *  @param chinfo 渠道链 需要传入JSON字典格式字符串！
 *
 */
+ (void)updateChinfo:(NSString *)chinfo;

/**
 * 渠道链UUID
 * 为了避免渠道链过长，生成一条UUID和chinfo对应的埋点记录当前的渠道链
 *
 * @param extParams 自定义扩展字段
 * @return UUID
 */
+ (nullable NSString*)chinfoChainUUIDWithBizName:(nullable NSString *)bizName;

/**
 * 渠道链UUID
 * 为了避免渠道链过长，生成一条UUID和chinfo对应的埋点记录当前的回退过滤的渠道链
 *
 * @param extParams 自定义扩展字段
 * @return UUID
 */
+ (nullable NSString*)backFilteredChinfoChainUUIDWithBizName:(nullable NSString *)bizName;

/**
 * 返回 过滤回退渠道链 中的首个渠道 {"newChinfo" : ch_xxx, "scm" : x.xxx.xxx}
 */
+ (nullable NSString*)firstChinfo;

#pragma mark - 渠道

/**
 * 设置下一个页面newChinfo和scm
 * @param newChinfo 渠道
 * @param scm
 */
+ (void)setNextPageNewChinfo:(NSString *)newChinfo
                         scm:(NSString *)scm;

/**
 * 设置当前页面newChinfo和scm
 * @param newChinfo 渠道
 * @param scm
 * @param view 需要和当前页面view相同
 */
+ (void)setCurrentPageNewChinfo:(NSString *)newChinfo
                            scm:(NSString *)scm
                           view:(NSObject *)view;

#pragma mark - 扩展参数
/**
 * 设置到下一个页面（非回退，非重复打开，非刷新）扩展4
 * @param dict 扩展参数字典，添加到扩展4 时，如果有key冲突，这个dict里对应的value会被丢掉
 * 注： 同一个页面打开之前，多次调用，dict会进行合并更新，后调用传入的dict key优先级更高
 */
+ (void)setNextPageExtParams4:(NSDictionary *)dict;

#pragma mark - 圣火令

/**
* 获取gPathId 并埋一个事件埋点
* @param bizName 业务名称 推荐使用 埋点的bizcode
*
*/
+ (nullable NSString *)torchGuidePathIdForBizName:(NSString *)bizName;

/**
* 获取gPathId 并埋一个事件埋点
* @param convType 转换类型
* @param extParams 扩展参数
*/
+ (nullable NSString *)torchGuidePathIdForConvType:(NSString *)convType extParams:(NSDictionary *)extParams;

/**
* 获取当前已经记录的 gPath
* 注: 因为调用时机不确定，可能会因为当前用户行为未记录完成，丢失最近的一些用户行为路径记录
*/
+ (nullable NSString*)currentTorchGuidePath;

/**
 * 同步获取 gPath
 * 注: 因为调用时机不确定，可能会因为当前用户行为未记录完成，会有一点轻微阻塞时间
 */
+ (nullable NSString *)syncGetTorchGuidePath;

/**
 * 异步获取 gPath
 */
+ (void)asyncGetTorchGuidePath:(void (^)(NSString * _Nullable))gpathBlock;

+ (NSString *)currentStepTorchGuidePath;

+ (void)torchConvEventWitBizName:(NSString *)bizName
                        convType:(nullable NSString *)convType
                       extParams:(NSDictionary *)extParams;

@end

NS_ASSUME_NONNULL_END

#endif
