//
//  APMInline.h
//  APMediaEditor
//
//  Created by yangjianxing.yjx on 2021/6/30.
//  Copyright © 2021 Alipay. All rights reserved.
//

#import <Foundation/Foundation.h>
#import <UIKit/UIKit.h>

NS_ASSUME_NONNULL_BEGIN

#define APM_ALWAYS_INLINE __inline__ __attribute__((always_inline))

#pragma mark - 判断实例类型

/// 判空
/// @param obj 传入的实例
/// @return 是否是 nil 或 NSNull
static APM_ALWAYS_INLINE BOOL apm_isNullOrNil(id _Nullable obj) {
    return !obj || [obj isKindOfClass:[NSNull class]];
}

/// 是否 NSString
/// @param obj obj 传入的实例
/// @return 是否 NSString
static APM_ALWAYS_INLINE BOOL apm_isStr(id _Nullable obj) {
    return [obj isKindOfClass:[NSString class]];
}

/// 是否 NSNumber
/// @param obj 传入的实例
/// @return 是否 NSNumber
static APM_ALWAYS_INLINE BOOL apm_isNum(id _Nullable obj) {
    return [obj isKindOfClass:[NSNumber class]];
}

/// 是否 NSString 或 NSNumber
/// @param obj 传入的实例
/// @return 是否是 NSString 或 NSNumber
static APM_ALWAYS_INLINE BOOL apm_isStrOrNum(id _Nullable obj) {
    return apm_isStr(obj) || apm_isNum(obj);
}

/// 是否 NSDictionary
/// @param obj 传入的实例
/// @return 是否 NSDictionary
static APM_ALWAYS_INLINE BOOL apm_isDic(id _Nullable obj) {
    return [obj isKindOfClass:[NSDictionary class]];
}

/// 是否 NSArray
/// @param obj 传入的实例
/// @return 是否 NSArray
static APM_ALWAYS_INLINE BOOL apm_isArr(id _Nullable obj) {
    return [obj isKindOfClass:[NSArray class]];
}

#pragma mark - 转换

/// 转 NSString
/// @param value 传入的实例
/// @return 转 NSString
static APM_ALWAYS_INLINE NSString * _Nullable apm_str(id _Nullable value) {
    return apm_isStr(value) ? value : (apm_isNum(value) ? ((NSNumber *)(value)).stringValue : nil);
}

/// 2个字符串拼接 (可传入 NSString NSNumber)
/// @param value0 参数1 <NSString or NSNumber>
/// @param value1 参数2 <NSString or NSNumber>
/// @return 返回拼接后的字符串
static APM_ALWAYS_INLINE NSString * _Nullable apm_strCat2(id _Nullable value0, id _Nullable value1) {
    NSString *str0 = apm_str(value0);
    NSString *str1 = apm_str(value1);
    if (!str0 && !str1) {
        return nil;
    }
    else {
        return [NSString stringWithFormat:@"%@%@", str0 ? str0 : @"", str1 ? str1 : @""];
    }
}

/// 3个字符串拼接 (可传入 NSString NSNumber)
/// @param value0 参数1 <NSString or NSNumber>
/// @param value1 参数2 <NSString or NSNumber>
/// @param value2 参数3 <NSString or NSNumber>
/// @return 返回拼接后的字符串
static APM_ALWAYS_INLINE NSString * _Nullable apm_strCat3(id _Nullable value0, id _Nullable value1, id _Nullable value2) {
    return apm_strCat2(apm_strCat2(value0, value1), value2);
}

/// 转 NSInteger
/// @param value 传入参数 <NSString or NSNumber>
/// @return 返回 NSInteger
static APM_ALWAYS_INLINE NSInteger apm_int(id _Nullable value) {
    return apm_isStrOrNum(value) ? [value integerValue] : 0;
}

/// 转 NSUInteger, value<0 则返回 0
/// @param value 传入参数 <NSString or NSNumber>
/// @return 返回 NSUInteger
static APM_ALWAYS_INLINE NSUInteger apm_uInt(id _Nullable value) {
    NSInteger num = apm_int(value); return num < 0 ? 0 : num;
}

/// 转 long long
/// @param value 传入参数 <NSString or NSNumber>
/// @return 返回 long long
static APM_ALWAYS_INLINE long long apm_longlong(id _Nullable value) {
    return apm_isStrOrNum(value) ? [value longLongValue] : 0;
}

/// 转 float
/// @param value 传入参数 <NSString or NSNumber>
/// @return 返回 float
static APM_ALWAYS_INLINE float apm_float(id _Nullable value) {
    return apm_isStrOrNum(value) ? [value floatValue] : 0.0;
}

/// 转 double
/// @param value 传入参数 <NSString or NSNumber>
/// @return 返回 double
static APM_ALWAYS_INLINE double apm_double(id _Nullable value) {
    return apm_isStrOrNum(value) ? [value doubleValue] : 0.0;
}

/// 转 BOOL
/// @param value 传入参数 <NSString or NSNumber>
/// @return 返回 BOOL
static APM_ALWAYS_INLINE BOOL apm_bool(id _Nullable value) {
    return apm_isStrOrNum(value) ? [value boolValue] : NO;
}

/// 转 NSDictionary
/// @param value 传入参数
/// @return 返回 NSDictionary
static APM_ALWAYS_INLINE NSDictionary * _Nullable apm_dic(id _Nullable value) {
    return apm_isDic(value) ? value : nil;
}

/// 转 NSArray
/// @param value 传入参数
/// @return 返回 NSArray
static APM_ALWAYS_INLINE NSArray * _Nullable apm_arr(id _Nullable value) {
    return apm_isArr(value) ? value : nil;
}

#undef APM_ALWAYS_INLINE

NS_ASSUME_NONNULL_END
