//
// ImageLoader class
// Utils function and classes for image loading.
// Created by tyreal.yj on 2020/09/17.
//

#ifndef XREAL_IMAGE_LOADER_H
#define XREAL_IMAGE_LOADER_H

#include <string>

namespace multimedia {
    class ImageLoader;

    /**
     * Image holder, holds image data
     */
    class ImageHolder {
        friend class ImageLoader;

    public:
        enum Format {
            RGB,
            RGBA,
        };

        ~ImageHolder();

        inline unsigned int getWidth() const { return mWidth; };

        inline unsigned int getHeight() const { return mHeight; };

        inline Format getFormat() const { return mFormat; }

        inline unsigned char *getData() const { return mData; };

    protected:
        ImageHolder();

        ImageHolder(unsigned int w, unsigned int h, Format format, unsigned char *data);

    private:
        unsigned int mWidth;
        unsigned int mHeight;
        Format mFormat;
        unsigned char *mData;
    };

    /**
     * Image Loader, supports jpeg/png format
     */
    class ImageLoader {
    public:
        typedef struct _callback {
            int  (*read) (void *stream, char *data, int size);   // fill 'data' with 'size' bytes. return number of bytes actually read
            void (*skip) (void *stream, int n);                  // skip the next 'n' bytes, or 'unget' the last -n bytes if negative
            int  (*eof)  (void *stream);                         // returns nonzero if we are at end of file/data
        } IOCallback;

        /**
         * Get image width and height info from file
         * @param path image file path
         * @param w width of image
         * @param h height of image
         * @return result, true or false
         */
        static bool getImageInfo(const std::string &path, int &w, int &h);

        /**
         * Get image width and height info from buffer (memory)
         * @param dataBuffer image data buffer
         * @param length buffer length
         * @param w width of image
         * @param h height of image
         * @return result, true or false
         */
        static bool getImageInfo(const unsigned char *dataBuffer, int length, int &w, int &h);

        /**
         * Get image width and height info from stream with callback function
         * @param stream image stream
         * @param callback stream IO callback function
         * @param w width of image
         * @param h height of image
         * @return result, true or false
         */
        static bool getImageInfo(void *stream, const IOCallback &callback, int &w, int &h);

        /**
         * load image from file path
         * @param path image file path
         * @param flip need image flipping or not, 1/0
         * @return image, return nullptr if load failed
         */
        static ImageHolder *loadImage(const std::string &path, int flip = 1);

        /**
         * load image from buffer
         * @param dataBuffer image buffer
         * @param length buffer length
         * @param flip need image flipping or not, 1/0
         * @return image, return nullptr if load failed
         */
        static ImageHolder *loadImage(const unsigned char *dataBuffer, int length, int flip = 1);

        /**
         * load image from stream with callback function
         * @param stream image stream
         * @param callback stream IO callback function
         * @param flip need image flipping or not, 1/0
         * @return image, return nullptr if load failed
         */
        static ImageHolder *loadImage(void *stream, const IOCallback &callback, int flip = 1);
    };
}

#endif
