/*
 * encoding: utf-8 
 * author: 胡军伟（苍茫）072224 
 * content: 面向ObjC的接口 
 * history: 2017-02-07 初次完成  
 */
#ifndef COM_ALIPAY_BIFROST
#define COM_ALIPAY_BIFROST
/*############################################################################*/
#include <stddef.h>
#include <stdbool.h>
#include <inttypes.h>
/*||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||||*/
typedef enum BIFROST_state {
    BIFROST_state_broken,      //连接断开。 
    BIFROST_state_linking,     //正在连接。 
    BIFROST_state_connected,   //已经连上。 
    BIFROST_state_handshaking, //正在进行SSL握手。 
    BIFROST_state_established, //已经完成SSL握手。  
    BIFROST_state_shutting     //正在关闭SSL。 
} BIFROST_state_t;
/*============================================================================*/
typedef enum BIFROST_err {
    BIFROST_err_time_out = -1,//超时。 
    BIFROST_err_ok,           //没有错误。 
    BIFROST_err_address,      //DNS查询问题。 
    BIFROST_err_receive,      //接收发生问题。 
    BIFROST_err_send,         //发送发生问题。 
    BIFROST_err_system,       //调用系统API发生问题。 
    BIFROST_err_memory,       //内存分配失败。 
    BIFROST_err_link,         //Socket发生问题。 
    BIFROST_err_handshake,    //SSL握手发生问题。 
    BIFROST_err_ssl_io,       //SSL收发发生问题。 
    BIFROST_err_shutdown,     //SSL关闭发生问题。 
    BIFROST_err_http,         //HTTP协议处理发生问题。 
    BIFROST_err_mmtp,         //MMTP协议处理发生问题。 
    BIFROST_err_init_msg = 67 //INIT关键信息中有空值。 
} BIFROST_err_t;
/*============================================================================*/
typedef enum BIFROST_alt {
    BIFROST_alt_net,       //表示网络状态切换。 
    BIFROST_alt_ground,    //表示前后台切换。 
    BIFROST_alt_screen,    //表示屏幕开关切换。 
    BIFROST_alt_process,   //表示主业务进程是否存在。 
    BIFROST_alt_user,      //表示用户登入登出行为。 
    BIFROST_alt_heartbeat, //表示心跳模式。 
    BIFROST_alt_security,  //表示安全模式。 
    BIFROST_alt_rtt,       //表示SSL握手模式。 
    BIFROST_alt_address,   //表示地址列表更新方式。 
    BIFROST_alt_single,    //表示单发建连失败多少次后开始进行并发建连。 
    BIFROST_alt_huawei,    //表示心跳代管是否允许使用。 
    BIFROST_alt_scene,     //表示当前场景的变更情况（目前用于短频建链）。 
    BIFROST_alt_switch,    //表示开关有更新。 
    BIFROST_alt_conn_limit, //表示是否限制建连。 
    BIFROST_alt_resend_init, //表示重发init。 
    BIFROST_alt_resend_sessionid, //表示sessionid补偿。 
    BIFROST_alt_load_file  //表示可以读大文件了（因为启动的时候不允许做耗时动作）
} BIFROST_alt_t;
/*============================================================================*/
typedef enum BIFROST_target_type {
    BIFROST_target_mmtp,    //表示内部长连接使用mmtp协议。 
    BIFROST_target_http2,   //表示内部长连接使用http2协议。
    BIFROST_target_mqtt     //the longlink use mqtt protocol
} BIFROST_target_type_t;
typedef enum BIFROST_sub_target_type {
    BIFROST_sub_target_only_rpc = 1   //表示内部长连接仅支持RPC。
} BIFROST_sub_target_type_t;
/*============================================================================*/
//the same as the stn.h
//static const int32_t kStreamingClient = 0x1;
//static const int32_t kStreamingServer = 0x2;
//static const int32_t kStreamingBoth = 0x3;
typedef enum BIFROST_streaming_type {
    BIFROST_streaming_none,    //not stream rpc
    BIFROST_streaming_client,  //client stream
    BIFROST_streaming_server,  //server stream
    BIFROST_streaming_both     //both client and server stream
} BIFROST_streaming_type_t;

typedef enum BIFROST_stream_post_state {
    BIFROST_stream_post_state_start,
    BIFROST_stream_post_state_continue,
    BIFROST_stream_post_state_finish,
    BIFROST_stream_post_state_cancel
} BIFROST_stream_post_state_t;
/*============================================================================*/
typedef enum BIFROST_TLocalIPStack {
    BIFROST_ELocalIPStack_None = 0,
    BIFROST_ELocalIPStack_IPv4 = 1,
    BIFROST_ELocalIPStack_IPv6 = 2,
    BIFROST_ELocalIPStack_Dual = 3,
} BIFROST_TLocalIPStack_t;
/*============================================================================*/
typedef struct BIFROST_kv {
    char *key;
    char *val;
} BIFROST_kv_t;
/*============================================================================*/
typedef struct BIFROST_cfg {
    BIFROST_kv_t *kv;
    size_t        count; //kv指向的元素数。
} BIFROST_cfg_t;
/*============================================================================*/
typedef struct BIFROST_touching {
    bool yes_lnk; //是否连上过。 
    /*------------------------------------------------------------------------*/
    bool proxy; //是否通过代理建连。 
    /*------------------------------------------------------------------------*/
    bool tryIPv6;//是否尝试过IPv6建连
    /*------------------------------------------------------------------------*/
    //网络类型。 
    char *network;
    char *netname;
    /*------------------------------------------------------------------------*/
    //两端的地址信息。 
    char *ip_local;
    char *ip_remote; //可能是真实服务器，可能是代理。 
    char *ip_server; //在代理的情况下，填真实服务器。 
    int   port_local;
    int   port_remote;
    int   port_server;
    /*------------------------------------------------------------------------*/
    //各项耗时（毫秒）。 
    double ms_dns;     //DNS查询耗时。 
    double ms_tcp;     //TCP建连耗时。 
    double ms_attempt; //从上次断连后首次发起连接到这次连上的耗时。 
    /*------------------------------------------------------------------------*/
    long long id_conn; //连接id。 
    /*------------------------------------------------------------------------*/
    char * quic_connid; //quic connect id
    /*------------------------------------------------------------------------*/
    int cnt_attempt; //从上次断连后到这次连上的尝试次数。
    int cnt_ip_index;//TCP建链成功使用的第几个IP,用于TCP建连次数分布统计
    int reason;      //建连原因。 
    int target_type;//类型，mmtp连接还是http2连接。 
    int channel_selected;
    
    BIFROST_cfg_t conn_perf_items; //建连 埋点字段携带
} BIFROST_touching_t;
/*============================================================================*/
typedef struct BIFROST_keeping {
    long long stamp;    //发心跳的时间点（毫秒）。 
    int       interval; //距离上个心跳的间隔。 
    int       duration; //距离上个心跳的实际间隔。 
    double    rtt;      //本次心跳的RTT（毫秒）。 
} BIFROST_keeping_t;
/*============================================================================*/
typedef struct BIFROST_separating {
    //为true则fast、ticket、standard和ms_ssl有效。 
    bool yes_ssl; //是否启用了SSL。 
    /*------------------------------------------------------------------------*/
    //为true则code_err和inf_err有效。 
    bool yes_err; //是否发生了错误。 
    /*------------------------------------------------------------------------*/
    bool fast;     //是否使用了0-RTT。 
    bool ticket;   //是否使用了ticket。 
    bool standard; //是否用了标准SSL协议。
    bool tlsv1_3;
    bool resumption;
    bool ssl_done; //ssl has handshake ok or not
    /*------------------------------------------------------------------------*/
    bool tryIPv6;//是否尝试过IPv6建连
    /*------------------------------------------------------------------------*/
    //各项耗时（毫秒）。 
    double ms_ssl;   //SSL握手耗时。 
    double ms_life;  //连接保持时长。 
    double ms_first; //从上次断连后首次发起连接到收到首帧的耗时。 
    /*------------------------------------------------------------------------*/
    int code_err; //错误码。 
    int target_type;//类型，mmtp连接还是http2连接。 
    int channel_selected;
    /*------------------------------------------------------------------------*/
    char *mtag;    //服务端从init回包中下发的策略埋点。 
    char *inf_err; //错误信息。 
    char *inf_crt; //扩展信息，比如握手异常时的证书信息等。 
    /*------------------------------------------------------------------------*/
    BIFROST_keeping_t *inf_hb; //智能心跳信息。 
    /*------------------------------------------------------------------------*/
    size_t num_hb; //智能心跳信息的个数。
    
    BIFROST_cfg_t conn_perf_items; //断连 埋点字段携带
} BIFROST_separating_t;
/*============================================================================*/
typedef struct BIFROST_bin {
    void  *bin;
    size_t len; //bin指向的数据的字节数。 
} BIFROST_bin_t;
/*============================================================================*/
typedef struct BIFROST_kv_bin {
    char *key;
    BIFROST_bin_t val;
} BIFROST_kv_bin_t;
/*============================================================================*/
typedef struct BIFROST_bin_cfg {
    BIFROST_kv_bin_t *bin_kv;
    size_t        count; //kv指向的元素数。 
} BIFROST_bin_cfg_t;
/*============================================================================*/
typedef struct BIFROST_swch {
    char *key;
    int   val;
} BIFROST_swch_t;
/*============================================================================*/
typedef struct BIFROST_collecting {
    int           channel;
    BIFROST_cfg_t data;
} BIFROST_collecting_t;
/*============================================================================*/
typedef struct BIFROST_gathering {
    int           channel;
    BIFROST_bin_t data;
} BIFROST_gathering_t;
/*============================================================================*/
typedef struct BIFROST_inf {
    char *major;
    char *minor;
} BIFROST_inf_t;
/*============================================================================*/
typedef struct BIFROST_init_rsp {
    int       status;
    int       reconn;
    long long origin;
    bool      ncrmnt;
    bool      makeup;
    char     *expand; //扩展MEMO，可以为NULL。 
    char     *device; //即MMTP-DID，可以为NULL。 
    /*------------------------------------------------------------------------*/
    char *mtag;      //策略埋点。 
    char *ip_client; //客户端出口IP。 
    char *id_dict;   //字典ID。 
    bool  user_io;   //登录登出事件是否上报。 
    bool  empty;     //init回包为空。 
} BIFROST_init_rsp_t;
/*============================================================================*/
typedef struct BIFROST_pkg {
    BIFROST_cfg_t header;
    BIFROST_bin_t body;
} BIFROST_pkg_t;
/*============================================================================*/
typedef struct BIFROST_obtaining {
    BIFROST_pkg_t pkg;
    /*------------------------------------------------------------------------*/
    long long receipt; //对于RPC就是RPC-ID。 
    int       channel;
    bool      retried; //是否重试过。 
    bool      oneshot; //是否在短连接上收发。 
    bool      use_short; //是否使用过短链接。
    bool      try_ipv6; //是否尝试过IPv6。
    bool      retry_pending;//是否有retryWithPending。
    /*------------------------------------------------------------------------*/
    //ms_caller减去ms_calling的差是从上层提交起，到实际发送的时间。 
    double ms_call; //从上层提交起，到本层收到的时间（毫秒）。 
    double ms_caller; //从上层提交请求起，到上层收到回复的时间（毫秒）。 
    double ms_calling; //从实际发送请求起，到上层收到回复的时间（毫秒）。 
    double ms_confirm; //从发送起到被spanner确认的时间（毫秒）。 
    /*------------------------------------------------------------------------*/
    double ms_air;
    double ms_amnet_all;
    double ms_queue_out;
    double ms_hung;
    double ms_queue_store;
    /*------------------------------------------------------------------------*/
    int len_pkg; //解压前的大小，负数表示无效。 
    int len_raw; //解压后的大小，负数表示无效。 
    int zip_req; //上行包的压缩方式。 
    int zip_rsp; //下行包的压缩方式。 
    /*------------------------------------------------------------------------*/
    int target_type;//数据包使用何种协议发送。mmtp=0,http2=1, mqtt=2
    int ip_stack;
    char *gateway6;
    /*------------------------------------------------------------------------*/
    char *mtag; //服务端从init回包中下发的策略埋点。 
    char *target_host; //连接的地址。 
    char *target_host_short; //短链接连接的地址。
    char *quic_conn_id;//Quic 连接 id

    /**
     * 该任务在Quic连接迁移中的状态. @see com::alipay::bifrost::task_ctx::TaskQuicConnMigrateStateEnum
     * miss_conn_migrate = 0,
     * hit_conn_migrate,
     * rsp_by_conn_migrate
     */
    int  task_quic_conn_migrate_state;
    /*------------------------------------------------------------------------*/
    long long id_conn; //连接id。

    int32_t   _previous_err_type;
    int32_t   _previous_err_code;
    bool      _is_sos_retried; //is sos tried.
    int32_t   _retry_count;    //重试次数

    BIFROST_streaming_type_t      streaming;         //see BIFROST_streaming_type_t
    BIFROST_stream_post_state_t   stream_rsp_state;  //see BIFROST_stream_post_state_t

    BIFROST_cfg_t rpc_perf_items; //RPC 埋点字段携带

} BIFROST_obtaining_t;
/*============================================================================*/
//用于异步处理完后的结果通知。 
//arg参数在回调时会填入本结构体对象的地址。 
typedef struct BIFROST_result {
    void (*notify)(void *arg, bool result);
} BIFROST_result_t;
/*============================================================================*/
//用于注册数据存取的回调函数。 
//每个函数的arg参数，在回调时会填入本结构体对象的地址。 
typedef struct BIFROST_storage {
    //从普通介质中获取。
    //返回取到的值，若不为NULL，则将字节数填入(*len)中。 
    //返回值必须可以使用free函数来释放。 
    void *(*get_common)(void *arg, const char *key, size_t *len);
    /*------------------------------------------------------------------------*/
    //从安全介质（比如黑匣子）中获取。 
    //返回取到的值，若不为NULL，则将字节数填入(*len)中。 
    //返回值必须可以使用free函数来释放。 
    void *(*get_secure)(void *arg, const char *key, size_t *len);
    /*------------------------------------------------------------------------*/
    //以同步方式保存到普通介质中。 
    //len是val指向的值的字节数。 
    void (*put_common)(void *arg, const char *key, const void *val, size_t len);
    /*------------------------------------------------------------------------*/
    //以同步方式保存到安全介质（比如黑匣子）中。 
    //len是val指向的值的字节数。 
    void (*put_secure)(void *arg, const char *key, const void *val, size_t len);
    /*------------------------------------------------------------------------*/
    //以异步方式保存到普通介质中。 
    //len是val指向的值的字节数。 
    void (*put_common_async)(void       *arg,
                             const char *key,
                             const void *val,
                             size_t      len);
    /*------------------------------------------------------------------------*/
    //以异步方式保存到安全介质（比如黑匣子）中。 
    //len是val指向的值的字节数。 
    void (*put_secure_async)(void       *arg,
                             const char *key,
                             const void *val,
                             size_t      len);
    /*------------------------------------------------------------------------*/
    //暂存准备保存到普通介质中的数据。 
    //len是val指向的值的字节数。 
    void (*cache_common)(void       *arg,
                         const char *key,
                         const void *val,
                         size_t      len);
    /*------------------------------------------------------------------------*/
    //暂存准备保存到安全介质（比如黑匣子）中的数据。 
    //len是val指向的值的字节数。 
    void (*cache_secure)(void       *arg,
                         const char *key,
                         const void *val,
                         size_t      len);
    /*------------------------------------------------------------------------*/
    //将暂存的数据以异步方式保存到对应介质中。 
    void (*commit)(void *arg);
    /*------------------------------------------------------------------------*/
    //从普通介质中删除。 
    void (*del_common)(void *arg, const char *key);
    /*------------------------------------------------------------------------*/
    //获取文件数据。
    void *(*get_big_data)(void *arg, const char *key, size_t *len, bool common);
    /*------------------------------------------------------------------------*/
    //存储大数据，一般是要单独存放文件，key为文件名
    void (*put_big_data_async)(void       *arg,
                               const char *key,
                               const void *val,
                               size_t      len,
                               bool        common);
    /*------------------------------------------------------------------------*/
    //删除大数据文件。
    void (*del_big_data)(void *arg, const char *key);
} BIFROST_storage_t;
/*============================================================================*/
//用于注册连接相关事件的处理函数。 
//每个函数的arg参数，在回调时会填入本结构体对象的地址。 
typedef struct BIFROST_linkage {
    //连接状态变化时的回调。 
    //state为新的状态，取值在BIFROST_state_t定义范围内。 
    void (*change)(void *arg, int state);
    /*------------------------------------------------------------------------*/
    //连接发生错误时的回调。 
    //err为错误类型，取值在BIFROST_err_t定义范围内。 
    //inf为描述。 
    /*
    void (*panic)(void *arg, int err, const char *inf);
    */
    /*------------------------------------------------------------------------*/
    //连接完成时的回调。 
    void (*touch)(void *arg, const BIFROST_touching_t *rpt);
    /*------------------------------------------------------------------------*/
    //心跳一定次数后的回调。 
    void (*keep)(void                     *arg,
                 const BIFROST_touching_t *rpt_touch,
                 const BIFROST_keeping_t  *rpt_hb,
                 size_t                    cnt_hb);
    /*------------------------------------------------------------------------*/
    //连接断开时的回调。 
    void (*separate)(void                       *arg,
                     const BIFROST_touching_t   *rpt_touch,
                     const BIFROST_separating_t *rpt_separate);
    /*------------------------------------------------------------------------*/
    //切到前台时报告是否有现成连接的回调。 
    //rpt为NULL表示没有现成连接。 
    /*
    void (*retrench)(void *arg, const BIFROST_touching_t *rpt);
    */
    /*------------------------------------------------------------------------*/
    //准备向服务器发送初始化请求时的回调。 
    //收集各channel的init数据。 
    //未经序列化的填入(*value_0)和(*count_0)中。 
    //序列化过的填入(*value_1)和(*count_1)中。 
    //输出的(*value_0)和(*value_1)不该为NULL， 
    //且指向的元素中的各级指针必须都可以使用free函数来释放。  
    void (*gain)(void                  *arg,
                 BIFROST_collecting_t **value_0,
                 size_t                *count_0,
                 BIFROST_gathering_t  **value_1,
                 size_t                *count_1);
    /*------------------------------------------------------------------------*/
    //埋点日志上报。 
    //type为埋点日志类型。 
    //param为埋点日志类型的版本号，可以为NULL。 
    /*
    void (*event_track)(void                *arg,
                        const char          *type,
                        const char          *param,
                        const BIFROST_cfg_t *data);
    */
    /*------------------------------------------------------------------------*/
    //收到初始化响应时的回调。 
    void (*establish)(void *arg);
    /*------------------------------------------------------------------------*/
    //要求重新登录时的回调。 
    void (*reactivate)(void *arg);
    /*------------------------------------------------------------------------*/
    //被流控时的回调。 
    //delay为多少秒后将重连。 
    //inf为用JSON编码的附加信息。 
    void (*server_restrict)(void *arg, int delay, const char *inf);
    /*------------------------------------------------------------------------*/
    //统计量上报。 
    //key在各BIFROST_rpt_*的定义范围内。 
    /*
    void (*report)(void *arg, const char *key, double val);
    */
    /*------------------------------------------------------------------------*/
    //通知上层一些关键值的情况。 
    //key在各BIFROST_gift_key_*的定义范围内。 
    //val在各BIFROST_gift_val_*的定义范围内。 
    /*
    void (*gift)(void *arg, const char *key, const char *val);
    */
    /*------------------------------------------------------------------------*/
    //额外配置上报。  
    void (*command)(void *arg, const BIFROST_bin_t *bin);
    /*------------------------------------------------------------------------*/
    //上层提交的数据超时未被确认时的回调。 
    //receipt是被发数据的唯一标识（对于RPC就是RPD-ID）。 
    //err为错误类型，在BIFROST_err_t定义范围内；-1表示无错误。 
    //inf为描述。
    //data为map,用于传递扩展信息
    void (*sorry)(void *arg, long long receipt, int err, const char *inf,
                  int target_type, const BIFROST_cfg_t *data);
    /*------------------------------------------------------------------------*/
    //向上层查询当前网络类别时的回调。 
    //网络类别填入(inf->major)，可以复制BIFROST_net_*，也可以自定义。 
    //网络的名字填入(inf->minor)。 
    //inf中各指针必须都可以使用free函数来释放。 
    /*
    void (*network)(void *arg, BIFROST_inf_t *inf);
    */
    /*------------------------------------------------------------------------*/
    //开启系统时钟。 
    //在ms毫秒后唤醒。 
    /*
    bool (*start_alarm)(void *arg, int id, long long ms);
    */
    /*------------------------------------------------------------------------*/
    //取消系统时钟。 
    /*
    bool (*stop_alarm)(void *arg, int id);
    */
    /*------------------------------------------------------------------------*/
    //init响应传给上层。  
    void (*init_rsp)(void *arg, const BIFROST_init_rsp_t *inf);
    /*------------------------------------------------------------------------*/
    /**
     * channel_select value :
     * static const int kChannelShort      = 0x1;
     * static const int kChannelLong       = 0x2;
     * static const int kChannelBoth       = 0x3;
     * static const int kChannelMultiplex  = 0x4;
     * static const int kChannelQuicMultiplex = 0x8;
     * static const int kChannelAll        = 0xF; //all of channel
     */
    bool (*is_connection_limited)(void *arg, int32_t channel_select);
    /*------------------------------------------------------------------------*/
    //要求补偿sessionid时的回调.  
    void (*resend_sessionid)(void *arg);
    /*------------------------------------------------------------------------*/
    //上报埋点信息 
    void (*report_perf_info)(void *arg, const BIFROST_cfg_t *data);
    /*------------------------------------------------------------------------*/
    //上层提交的数据发送进度通知。
    //receipt是被发数据的唯一标识（对于RPC就是RPD-ID）。
    //progress为当前进度，取值范围0-100
    void (*progress_update)(void *arg, long long receipt, int progress);
    /*------------------------------------------------------------------------*/
} BIFROST_linkage_t;
/*============================================================================*/
typedef struct BIFROST_mqtt_callback {
    //the return value(void *) must can be free outside
    //the len praram set the length of void* data
    void* (*mqtt_connect_msg)(void *arg, size_t *len);
    /*------------------------------------------------------------------------*/
    //the return value(void *) must can be free outside
    //the len praram set the length of void* data
    void* (*mqtt_hb_req_msg)(void *arg, size_t *len);
    /*------------------------------------------------------------------------*/
    //return value: >0 is the decode length, 0 is to continue to read, -1 is error
    //ack_msgid is the msgid of the server response the client request
    //type is the decode frame type
    //notice: one time decode one frame
    int (*mqtt_decode)(void             *arg,
                       const uint8_t    *data,
                       size_t            size,
                       int              *ack_msgid,
                       int              *type);
    /*------------------------------------------------------------------------*/
    void (*mqtt_data_result)(void          *arg,
                             long long      pkg_id,
                             int            err_code,
                             const char    *err_msg);
    /*------------------------------------------------------------------------*/
} BIFROST_mqtt_callback_t;
/*============================================================================*/
//每个函数的arg参数，在回调时会填入本结构体对象的地址。 
typedef struct BIFROST_channel {
    //收到数据时的回调。 
    void (*obtain)(void *arg, const BIFROST_obtaining_t *inf);
    /*------------------------------------------------------------------------*/
    //获取receipt时的回调。 
    //若无receipt，返回负数即可。 
    /*
    long long (*earnest)(void *arg, const BIFROST_pkg_t *pkg);
    */
    /*------------------------------------------------------------------------*/
    //获取traceid时的回调。 
    //若无traceid，返回NULL即可。 
    /*
    const char *(*tracer)(void *arg, const BIFROST_pkg_t *pkg);
    */
    /*------------------------------------------------------------------------*/
    //数据发出后的回调。 
    //receipt是被发数据的唯一标识（对于RPC就是RPD-ID）。 
    //len_raw是压缩前的大小。 
    //len_pkg是压缩后的大小。 
    void (*tell)(void *arg, long long receipt, size_t len_raw, size_t len_pkg);
    /*------------------------------------------------------------------------*/
    //回收发不出去的数据时的回调。 
    void (*recycle)(void *arg, const BIFROST_pkg_t *pkg);
} BIFROST_channel_t;
/*============================================================================*/
//用于注册某channel的处理函数。 
typedef struct BIFROST_service {
    int                channel;
    BIFROST_channel_t *callback;
} BIFROST_service_t;
/*============================================================================*/
//用于注册公用的回调函数。 
//每个函数的arg参数，在回调时会填入本结构体对象的地址。 
typedef struct BIFROST_registering {
    //记录日志的函数。 
    void (*record)(void       *arg,
                   const char *level,
                   const char *tag,
                   const char *content);
    /*------------------------------------------------------------------------*/
    //get the proxy info
    //host: the host to be proxy
    //if has proxy info, the (*host_proxy) must not be null, and can be free
    //if has username info, the (*username) must not be null, and can be free
    //if has password info, the (*password) must not be null, and can be free
    //proxy_type:
    //enum ProxyType {
    //    kProxyNone = 0,
    //    kProxyHttpTunel,
    //    kProxySocks5,
    //    kProxyHttp,
    //};
    bool (*app_proxy_info)(void         *arg,
                           const char   *host,
                           char        **host_proxy,
                           int          *port_proxy,
                           char        **username,
                           char        **password,
                           int          *proxy_type);
    /*------------------------------------------------------------------------*/
    //获取应用的路径。 
    //输出的(*path)不该为NULL，且指向的字符串以使用free函数来释放。 
    void (*app_path)(void *arg, char **path);
    /*------------------------------------------------------------------------*/
    //获取应用的路径。
    //输出的(*utdid)不该为NULL，且指向的字符串以使用free函数来释放。
    void (*app_utdid)(void *arg, char **utdid);
    /*------------------------------------------------------------------------*/
    //获取版本号信息。 
    unsigned int (*app_version)(void *arg);
    /*------------------------------------------------------------------------*/
    //获取账号信息。 
    //输出的(*name)不该为NULL，且指向的字符串以使用free函数来释放。 
    //返回是否登录。 
    bool (*account)(void *arg, char **name, long long *uin);
    /*------------------------------------------------------------------------*/
    //获取设备信息。 
    //输出的(*name)和(*type)不该为NULL，且指向的字符串以使用free函数来释放。 
    void (*device)(void *arg, char **name, char **type);
    /*------------------------------------------------------------------------*/
    //get httpdns ip
    //return the httpdns ips, and the ips_size is the total count of result ips
    char** (*new_dns)(void *arg, const char *host, int *ips_size);
    //clean httpdns ip for host
    //return the result
    bool (*clean_new_dns)(void *arg, const char *host);
    //on demand get httpdns ip
    //return the httpdns ips, and the ips_size is the total count of result ips
    char** (*on_demand_dns)(void *arg, const char *host, int *ips_size);
    /*------------------------------------------------------------------------*/
    //使用数据网卡的前置检查。
    bool (*bind_cellular_check)(void *arg, const char *host, int fd, const int index);
    /*------------------------------------------------------------------------*/
    //是否真正使用了数据网卡的回调
    void (*bind_cellular_notify)(void *arg, bool bind_result);
    //域名host是否支持国密双证书，lib_type为1表示STN，为2表示DTN
    bool (*is_host_support_use_gm)(void *arg, const char *host, const int lib_type);
} BIFROST_registering_t;
/*============================================================================*/
//用于设置开关。 
typedef struct BIFROST_switching {
    BIFROST_swch_t *swch;
    size_t          count_swch;
} BIFROST_switching_t;
/*============================================================================*/
//id是调用者为每个连接目标生成的唯一编号。 
//storage若为NULL则表示不变更回调逻辑。 
//linkage若为NULL则表示不变更回调逻辑。 
//swchmng若为NULL则表示不变更回调逻辑。 
//service若为NULL则表示所有channel都不变更回调逻辑。 
//service中某项callback为NULL的项表示注销，非NULL的项表示新增或变更。 
//ground为true表示主业务在前台，为false表示在后台。 
//screen为true表示亮屏，为false表示黑屏。 
//master为true表示主业务进程健在，为false表示已停止运行。 
typedef struct BIFROST_activating {
    long long id;
    /*------------------------------------------------------------------------*/
    int target_type;
    int sub_target_type;
    /*------------------------------------------------------------------------*/
    //BIFROST_storage_t *storage;
    BIFROST_linkage_t *linkage;
    BIFROST_mqtt_callback_t *mqtt_callback;
    /*------------------------------------------------------------------------*/
    BIFROST_service_t *service;
    size_t             count_service;
    /*------------------------------------------------------------------------*/
    char *domain_long;
    char *domain_short;
    char *domain_multiplex;
    char *domain_quic_multiplex;
    /*------------------------------------------------------------------------*/
    BIFROST_cfg_t cfg_long;
    BIFROST_cfg_t cfg_short;
    BIFROST_cfg_t cfg_multiplex;
    BIFROST_cfg_t cfg_quic_multiplex;
    BIFROST_cfg_t cfg_ext_params; //extra parameters
    /*------------------------------------------------------------------------*/
    BIFROST_inf_t net; //网络类型和名字。 
    /*------------------------------------------------------------------------*/
    bool ground;
    bool screen;
    bool master;
} BIFROST_activating_t;
/*============================================================================*/
typedef struct BIFROST_posting {
    long long id;
    /*------------------------------------------------------------------------*/
    long long stamp;     //时间戳；由统一库填入。 
    long long receipt;   //RPC用的；负数表示无效。 
    long long mqtt_pkg_id; //just for mqtt protocol, other set -1
    long long mqtt_msg_id; //just for mqtt protocol, other set -1; must has ack
    BIFROST_streaming_type_t      streaming;         //see BIFROST_streaming_type_t
    BIFROST_stream_post_state_t   stream_post_state; //see BIFROST_stream_post_state_t
    bool      important; //是否要服务端确认。 
    bool      secret;    //是否要加密。 
    bool      nearing;   //是否仅到spanner。 
    bool      shortcut;  //是否允许走短连接。 
    bool      nolong;    //是否不走长连接。
    char     *shortIPList;
    bool      multiplex;//是否走多路复用连接。
    bool      support_quic;     //是否支持quic
    bool      support_quic_backup_mode;
    bool      only_quic; //是否只走quic
    int       channel;
    char     *fwdip;     //SYNC用的。 
    char     *traceid;   //侦测日志用的。
    /*------------------------------------------------------------------------*/
    BIFROST_pkg_t pkg;
    BIFROST_cfg_t params; //扩展参数。 
} BIFROST_posting_t;
/*============================================================================*/
typedef struct BIFROST_altering {
    long long id;
    /*------------------------------------------------------------------------*/
    int status; //取值在BIFROST_alt_t定义范围内。 
    /*------------------------------------------------------------------------*/
    BIFROST_inf_t inf;
    BIFROST_bin_cfg_t bin_data;
} BIFROST_altering_t;
/*============================================================================*/
typedef struct BIFROST_alerting {
    long long id;
    /*------------------------------------------------------------------------*/
    long long receipt;  //不应为负数。 
    long long duration; //负数表示未超时，其他表示毫秒数。 
} BIFROST_alerting_t;
/*============================================================================*/
//每个函数的arg参数，在回调时会填入本结构体对象的地址。 
#if 0
typedef struct BIFROST_detecting {
    long long id;
    /*------------------------------------------------------------------------*/
    //fin为false，表示某连接刚连上，ok、done、summary无效。 
    //fin为true，表示某连接探测完成，此时再看另三个参数。 
    //ok表示某连接探测结果是否为成功。 
    //done表示本轮测试是否全部结束。 
    //summary记录了某连接的探测总结。 
    void (*report)(void       *arg,
                   bool        fin,
                   bool        ok,
                   bool        done,
                   const char *summary);
} BIFROST_detecting_t;
#endif
/*============================================================================*/
//每个函数的arg参数，在回调时会填入本结构体对象的地址。 
typedef struct BIFROST_asking {
    long long id;
    /*------------------------------------------------------------------------*/
    void (*answer)(void *arg, int state);
} BIFROST_asking_t;
/*============================================================================*/
//每个函数的arg参数，在回调时会填入本结构体对象的地址。
//channel_select的值：
//static const int kChannelMultiplex  = 0x4;
typedef struct BIFROST_preconnecting {
    long long id;
    /*------------------------------------------------------------------------*/
    char    *host;
    int      port;
    bool     ssl;
    int      channel_select;
} BIFROST_preconnecting_t;
/*============================================================================*/
#if defined(__cplusplus)
extern "C" {
#endif
/*++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++*/
//DNS查询时长（毫秒）。 
/*
extern const char *const BIFROST_rpt_dns;
*/
/*----------------------------------------------------------------------------*/
//TCP建连时长（毫秒）。 
/*
extern const char *const BIFROST_rpt_tcp_init;
*/
/*----------------------------------------------------------------------------*/
//SSL握手时长（毫秒）。 
/*
extern const char *const BIFROST_rpt_ssl_init;
*/
/*----------------------------------------------------------------------------*/
//发送一个MMTP帧的时长（毫秒）。 
/*
extern const char *const BIFROST_rpt_data_send;
*/
/*----------------------------------------------------------------------------*/
//连接保持时长（毫秒）。 
/*
extern const char *const BIFROST_rpt_link_life;
*/
/*----------------------------------------------------------------------------*/
//心跳RTT（毫秒）。 
/*
extern const char *const BIFROST_rpt_heartbeat_rtt;
*/
/*----------------------------------------------------------------------------*/
//从上次断连后首次发起连接到收到第一帧MMTP的时长（毫秒）。 
/*
extern const char *const BIFROST_rpt_first_frame;
*/
/*----------------------------------------------------------------------------*/
//从上次断连后首次发起连接到这次连上的时长（毫秒）。 
/*
extern const char *const BIFROST_rpt_attempt_duration;
*/
/*----------------------------------------------------------------------------*/
//从上次断连后到这次连上的尝试次数。 
/*
extern const char *const BIFROST_rpt_attempt_count;
*/
/*----------------------------------------------------------------------------*/
//是否使用0-RTT。 
/*
extern const char *const BIFROST_rpt_zero_rtt;
*/
/*----------------------------------------------------------------------------*/
//是否使用ticket机制。 
/*
extern const char *const BIFROST_rpt_ticket;
*/
/*----------------------------------------------------------------------------*/
//通知上层短链接模式是否开启。 
/*
extern const char *const BIFROST_gift_key_shortcut;
*/
/*----------------------------------------------------------------------------*/
//通知上层登入登出事件是否需要通过ST包上报。 
/*
extern const char *const BIFROST_gift_key_user_io;
*/
/*----------------------------------------------------------------------------*/
/*
extern const char *const BIFROST_gift_val_start;
extern const char *const BIFROST_gift_val_stop;
extern const char *const BIFROST_gift_val_true;
extern const char *const BIFROST_gift_val_false;
*/
/*----------------------------------------------------------------------------*/
extern void* BIFROST_dynamic_load_handler;
/*----------------------------------------------------------------------------*/
//用于配置安全协议。 
//若为null或者空字符串，则说明不启用SSL。 
//其他值在各BIFROST_ssl_*的定义范围内，超出范围算作BIFROST_ssl_dft。 
//若此项缺失，则算作不启用SSL。  
extern const char *const BIFROST_cfg_ssl __attribute ((visibility("default")));
extern const char* get_BIFROST_cfg_ssl();
/*----------------------------------------------------------------------------*/
//用于配置顶级CA证书。 
//若为null或者空字符串，则说明不进行证书链验证。 
//其他值都是PEM格式的证书文件内容。 
//若此项缺失，则算作不进行证书链验证。  
extern const char *const BIFROST_cfg_ca __attribute ((visibility("default")));
extern const char* get_BIFROST_cfg_ca();
/*----------------------------------------------------------------------------*/
//用于配置对端证书名。 
//若为null或者空字符串，则说明不进行证书名验证。 
//证书名是类似"amgw.alipay.com"的字符串。 
//若此项缺失，则算作不进行证书名验证。 
extern const char *const BIFROST_cfg_cert __attribute ((visibility("default")));
extern const char* get_BIFROST_cfg_cert();
/*----------------------------------------------------------------------------*/
extern const char *const BIFROST_check_cert __attribute ((visibility("default")));
extern const char* get_BIFROST_check_cert();
/*----------------------------------------------------------------------------*/
extern const char *const BIFROST_true __attribute ((visibility("default")));
extern const char* get_BIFROST_true();
extern const char *const BIFROST_false __attribute ((visibility("default")));
extern const char* get_BIFROST_false();
/*----------------------------------------------------------------------------*/
//用于配置服务侧地址列表。 
//若为null或者空字符串，则说明不使用地址列表。 
//多个地址用逗号隔开，比如"1.1.1.1:4444,2.2.2.2:55555"。 
//若此项缺失，则算作不使用地址列表。 
extern const char *const BIFROST_cfg_addr __attribute ((visibility("default")));
extern const char* get_BIFROST_cfg_addr();
/*----------------------------------------------------------------------------*/
extern const char *const BIFROST_ssl_mtls __attribute ((visibility("default"))); //使用MTLS协议。
extern const char * get_BIFROST_ssl_mtls();
extern const char *const BIFROST_ssl_std __attribute ((visibility("default")));  //用SSL库支持的最高版本协议。
extern const char * get_BIFROST_ssl_std();
extern const char *const BIFROST_ssl_dft __attribute ((visibility("default")));  //缺省协议。
extern const char * get_BIFROST_ssl_dft();
/*----------------------------------------------------------------------------*/
//当BIFROST_altering_t的status为BIFROST_alt_net时， 
//其inf的major取值范围如下： 
extern const char *const BIFROST_net_2g __attribute ((visibility("default")));      //表示当前为2G网络。
extern const char * get_BIFROST_net_2g();
extern const char *const BIFROST_net_3g __attribute ((visibility("default")));      //表示当前为3G网络。
extern const char * get_BIFROST_net_3g();
extern const char *const BIFROST_net_4g __attribute ((visibility("default")));      //表示当前为4G网络。
extern const char * get_BIFROST_net_4g();
extern const char *const BIFROST_net_wifi __attribute ((visibility("default")));    //表示当前为wifi网络。
extern const char * get_BIFROST_net_wifi();
extern const char *const BIFROST_net_unknown __attribute ((visibility("default"))); //表示当前为类型未知的网络。
extern const char * get_BIFROST_net_unknown();
extern const char *const BIFROST_net_none __attribute ((visibility("default")));    //表示当前无网络。
extern const char * get_BIFROST_net_none();
/*----------------------------------------------------------------------------*/
//当BIFROST_altering_t的status为BIFROST_alt_ground时， 
//其inf的major取值范围如下： 
extern const char *const BIFROST_ground_fore __attribute ((visibility("default"))); //表示当前在前台运行。
extern const char * get_BIFROST_ground_fore();
extern const char *const BIFROST_ground_back __attribute ((visibility("default"))); //表示当前在后台运行。
extern const char * get_BIFROST_ground_back();
/*----------------------------------------------------------------------------*/
//当BIFROST_altering_t的status为BIFROST_alt_screen时， 
//其inf的major取值范围如下： 
extern const char *const BIFROST_screen_i __attribute ((visibility("default"))); //表示亮屏。
extern const char * get_BIFROST_screen_i();
extern const char *const BIFROST_screen_o __attribute ((visibility("default"))); //表示锁屏。
extern const char * get_BIFROST_screen_o();
/*----------------------------------------------------------------------------*/
//当BIFROST_altering_t的status为BIFROST_alt_process时， 
//其inf的major取值范围如下： 
extern const char *const BIFROST_process_i __attribute ((visibility("default"))); //表示进程存在。
extern const char * get_BIFROST_process_i();
extern const char *const BIFROST_process_o __attribute ((visibility("default"))); //表示进程不在。
extern const char * get_BIFROST_process_o();
/*----------------------------------------------------------------------------*/
//当BIFROST_altering_t的status为BIFROST_alt_user时， 
//其inf的major取值范围如下： 
extern const char *const BIFROST_user_i __attribute ((visibility("default"))); //表示用户登入。
extern const char * get_BIFROST_user_i();
extern const char *const BIFROST_user_o __attribute ((visibility("default"))); //表示用户登出。
extern const char * get_BIFROST_user_o();
/*----------------------------------------------------------------------------*/
//当BIFROST_altering_t的status为BIFROST_alt_heartbeat时， 
//其inf的major取值范围如下： 
extern const char *const BIFROST_heartbeat_static __attribute ((visibility("default")));  //表示标准心跳。
extern const char * get_BIFROST_heartbeat_static();
extern const char *const BIFROST_heartbeat_dynamic __attribute ((visibility("default"))); //表示智能心跳。
extern const char * get_BIFROST_heartbeat_dynamic();
/*----------------------------------------------------------------------------*/
//当BIFROST_altering_t的status为BIFROST_alt_security时， 
//其inf的major取值范围如下： 
extern const char *const BIFROST_security_instant __attribute ((visibility("default"))); //表示全链加密。
extern const char * get_BIFROST_security_instant();
extern const char *const BIFROST_security_delayed __attribute ((visibility("default"))); //表示按需加密。
extern const char * get_BIFROST_security_delayed();
/*----------------------------------------------------------------------------*/
//当BIFROST_altering_t的status为BIFROST_alt_rtt时， 
//其inf的major取值范围如下： 
extern const char *const BIFROST_rtt_normal __attribute ((visibility("default"))); //表示正常的SSL握手。
extern const char * get_BIFROST_rtt_normal();
extern const char *const BIFROST_rtt_zero __attribute ((visibility("default")));   //表示0-RTT的SSL握手。
extern const char * get_BIFROST_rtt_zero();
/*----------------------------------------------------------------------------*/
//当BIFROST_altering_t的status为BIFROST_alt_address时， 
//其inf的major取值范围如下： 
extern const char *const BIFROST_address_all __attribute ((visibility("default")));   //表示更新所有地址列表。
extern const char * get_BIFROST_address_all();
extern const char *const BIFROST_address_long __attribute ((visibility("default")));  //表示只更新长连接地址列表。
extern const char * get_BIFROST_address_long();
extern const char *const BIFROST_address_short __attribute ((visibility("default"))); //表示只更新短连接地址列表。
extern const char * get_BIFROST_address_short();
extern const char *const BIFROST_address_multiplex __attribute ((visibility("default")));  //表示只更新多路复用连接地址列表。
extern const char * get_BIFROST_address_multiplex();
/*----------------------------------------------------------------------------*/
//当BIFROST_altering_t的status为BIFROST_alt_huawei时， 
//其inf的major取值范围如下： 
extern const char *const BIFROST_turn_on __attribute ((visibility("default")));  //表示开关开启。
extern const char * get_BIFROST_turn_on();
extern const char *const BIFROST_turn_off __attribute ((visibility("default"))); //表示开关关闭。
extern const char * get_BIFROST_turn_off();
/*----------------------------------------------------------------------------*/
//当BIFROST_altering_t的status为BIFROST_alt_conn_limit时， 
//其inf的major取值范围如下： 
extern const char *const BIFROST_conn_limit_on __attribute ((visibility("default")));   //限制建连。
extern const char * get_BIFROST_conn_limit_on();
extern const char *const BIFROST_conn_limit_off __attribute ((visibility("default")));  //不限制建连。
extern const char * get_BIFROST_conn_limit_off();
/*----------------------------------------------------------------------------*/
//当BIFROST_altering_t的status为BIFROST_alt_load_file时， 
//其inf的major取值范围如下 
extern const char *const BIFROST_load_file_on __attribute ((visibility("default"))); //告知可以加载文件
extern const char * get_BIFROST_load_file_on();
/*----------------------------------------------------------------------------*/
//此常量用于区分一般的数据包，如果扩展map里面包含此key，不管value的值，都会当做15号指令来处理。 
extern const char *const BIFROST_repost_msg __attribute ((visibility("default")));
extern const char * get_BIFROST_repost_msg();
//此常量用于传递数据的超时时间的key。目前只针对发消息RPC用到。value为超时时间，毫秒单位。 
extern const char *const BIFROST_data_timeout __attribute ((visibility("default")));
extern const char * get_BIFROST_data_timeout();
//此常量用于传递数据的后端系统id，同一channel下区分不同子级别系统。 
extern const char *const BIFROST_mmtp_sys_id __attribute ((visibility("default")));
extern const char * get_BIFROST_mmtp_sys_id();
extern const char *const BIFROST_mmtp_ext_echo_c __attribute ((visibility("default")));  //支持zstd时，数据header带的key
extern const char * get_BIFROST_mmtp_ext_echo_c();
extern const char *const BIFROST_val_support_zstd __attribute ((visibility("default"))); //支持zstd时，数据header带的val
extern const char * get_BIFROST_val_support_zstd();
//this used for the multiplexlink, set to the post_param kv
extern const char *const BIFROST_post_url __attribute ((visibility("default")));
extern const char * get_BIFROST_post_url();
/*----------------------------------------------------------------------------*/
//开关键值。
extern const char *const BIFROST_switch_imc __attribute ((visibility("default")));      //是否开启init增量。
extern const char * get_BIFROST_switch_imc();
extern const char *const BIFROST_switch_shortcut __attribute ((visibility("default"))); //是否开启短连接。
extern const char * get_BIFROST_switch_shortcut();
extern const char *const BIFROST_switch_zg __attribute ((visibility("default")));       //是否开启zstd
extern const char * get_BIFROST_switch_zg();
extern const char *const BIFROST_switch_zss __attribute ((visibility("default")));      //是否验证spanner支持zstd
extern const char * get_BIFROST_switch_zss();
extern const char *const BIFROST_switch_zdp __attribute ((visibility("default")));      //zstd降级时长,单位秒
extern const char * get_BIFROST_switch_zdp();
extern const char *const BIFROST_switch_shrink __attribute ((visibility("default"))); //spanner quickly shrink
extern const char * get_BIFROST_switch_shrink();
/*----------------------------------------------------------------------------*/
//这个函数必须最先调用。 
//返回true才可调用其他函数。 
__attribute ((visibility("default"))) bool BIFROST_init();
/*----------------------------------------------------------------------------*/
//destroy the bifrost lib, call this function when proc exit
__attribute ((visibility("default"))) void BIFROST_destroy();
/*----------------------------------------------------------------------------*/
//注册一些回调函数。 
//异步调用，有了结果后会回调cb中的notify。 
//cb可以为NULL，表示不关心结果。 
__attribute ((visibility("default"))) void BIFROST_register(const BIFROST_registering_t *param,
                      const BIFROST_result_t      *cb);
/*----------------------------------------------------------------------------*/
//设置存储回调函数。
//异步调用，有了结果后会回调cb中的notify。
//cb可以为NULL，表示不关心结果。
__attribute ((visibility("default"))) void BIFROST_register_storage(const BIFROST_storage_t *param,
                                                                    const BIFROST_result_t  *cb);
/*----------------------------------------------------------------------------*/
//设置开关。 
//异步调用，有了结果后会回调cb中的notify。 
//cb可以为NULL，表示不关心结果。 
__attribute ((visibility("default"))) void BIFROST_switch(const BIFROST_switching_t *param,
                    const BIFROST_result_t    *cb);
/*----------------------------------------------------------------------------*/
//启动一个AMNET。 
//异步调用，有了结果后会回调cb中的notify。 
//cb可以为NULL，表示不关心结果。 
__attribute ((visibility("default"))) void BIFROST_activate(const BIFROST_activating_t *param,
                      const BIFROST_result_t     *cb);
/*----------------------------------------------------------------------------*/
//发送一帧数据。 
//异步调用，有了结果后会回调cb中的notify。 
//cb可以为NULL，表示不关心结果。 
__attribute ((visibility("default"))) void BIFROST_post(const BIFROST_posting_t *param, const BIFROST_result_t *cb);
/*----------------------------------------------------------------------------*/
//环境变化通知。 
//异步调用，有了结果后会回调cb中的notify。 
//cb可以为NULL，表示不关心结果。 
__attribute ((visibility("default"))) void BIFROST_alter(const BIFROST_altering_t *param, const BIFROST_result_t *cb);
/*----------------------------------------------------------------------------*/
//RPC处理结果通知。 
//异步调用，有了结果后会回调cb中的notify。 
//cb可以为NULL，表示不关心结果。 
__attribute ((visibility("default"))) void BIFROST_alert(const BIFROST_alerting_t *param, const BIFROST_result_t *cb);
/*----------------------------------------------------------------------------*/
//探测MMTP链路是否好使。 
//异步调用，有了结果后会回调cb中的notify。 
//cb可以为NULL，表示不关心结果。 
/*
void BIFROST_detect(const BIFROST_detecting_t *param,
                    const BIFROST_result_t    *cb);
*/
/*----------------------------------------------------------------------------*/
//查询MMTP长连接链路状态。 
//异步调用，有了结果后会回调cb中的notify。 
//cb可以为NULL，表示不关心结果。 
__attribute ((visibility("default"))) void BIFROST_ask(const BIFROST_asking_t *param, const BIFROST_result_t *cb);
/*----------------------------------------------------------------------------*/
//停止一个AMNET。 
//异步调用，有了结果后会回调cb中的notify。 
//cb可以为NULL，表示不关心结果。 
__attribute ((visibility("default"))) void BIFROST_inactivate(long long id_, const BIFROST_result_t *cb);
/*----------------------------------------------------------------------------*/
//pre-connect the multiplexlink
__attribute ((visibility("default"))) void BIFROST_preconnect(const BIFROST_preconnecting_t *param, const BIFROST_result_t *cb);
/*----------------------------------------------------------------------------*/
//TLocalIPStack
__attribute ((visibility("default"))) BIFROST_TLocalIPStack_t BIFROST_local_ipstack_detect();
/*----------------------------------------------------------------------------*/
#if defined(__cplusplus)
}
#endif
/*############################################################################*/
#endif
