//
//  ASPEngineSDK.h
//  ASPEngineSDK
//
//  Created by 杨航涛 on 2021/10/25.
//

#import <Foundation/Foundation.h>
#import <UIKit/UIKit.h>

//! Project version number for ASPEngineSDK.
FOUNDATION_EXPORT double ASPEngineSDKVersionNumber;

//! Project version string for ASPEngineSDK.
FOUNDATION_EXPORT const unsigned char ASPEngineSDKVersionString[];

extern int ASP_MAX_MTS_POINTER;
extern float ASP_VIEW_MAX_SCALE;
extern float ASP_VIEW_MIN_SCALE;
extern float ASP_MOUSE_WHEEL_MIN_MOVE_DISTANCE;
extern float ASP_VIEW_MIN_SCALE_THRESHOLD;

#pragma mark - Data Structure

@interface ASPStatisticsInfo : NSObject

@property (nonatomic, assign) int mReceiveFps; // 端侧每秒接收到的视频祯数量
@property (nonatomic, assign) int mRenderFps; // 端侧渲染FPS
@property (nonatomic, assign) double mDownstreamBandwithMBPerSecond; // 下行带宽，MB/s
@property (nonatomic, assign) double mUpstreamBandwithMBPerSecond; // 上行带宽，MB/s
@property (nonatomic, assign) long mP2pFullLinkageLatencyMS; // 端到端全链路时延，毫秒
                                           // 目前该值需要在guest os中使用特定应用才可准确表示
@property (nonatomic, assign) long mNetworkLatencyMS; // 网络rtt时延，毫秒
@property (nonatomic, assign) long mPingGatewayRtt;   // Ping rtt时延，毫秒
@property (nonatomic, assign) double mLostRate; // 丢包率
@property (nonatomic, assign) long mServerRenderLatencyMS; // 服务端渲染时延，毫秒
@property (nonatomic, assign) long mServerEncoderLatencyMS; // 服务端编码时延，毫秒
@property (nonatomic, assign) long mServerTotalLatencyMS; // 服务端总时延，毫秒
@property (nonatomic, assign) long accumulateBandwidth; //总带宽
@property (nonatomic, copy  ) NSString *protocolType; //流协议类型

@property (nonatomic, copy  ) NSDictionary *infoDic;

@end

// PictureOrientation
typedef NS_ENUM(NSInteger, ASPPictureOrientation) {
    PictureOrientation_0 = 0,
    PictureOrientation_90 = 1,
    PictureOrientation_180 = 2,
};

// TouchAction
typedef NS_ENUM(NSInteger, ASPTouchAction) {
    TouchActionNone,
    TouchActionDown,
    TouchActionUp,
    TouchActionMove,
    TouchActionPointerDown,
    TouchActionPointerUp,
};

// MouseAction
typedef NS_ENUM(NSInteger, ASPMouseAction) {
    MouseActionNone,
    MouseActionDown,
    MouseActionUp,
};

// MouseButton
typedef NS_ENUM(NSInteger, ASPMouseButtonMask) {
    MouseButtonNone = 0,
    MouseButtonLeft = 1 << 0,
    MouseButtonMiddle = 1 << 1,
    MouseButtonRight = 1 << 2,
    MouseButtonLM = MouseButtonLeft | MouseButtonMiddle,
    MouseButtonLR = MouseButtonLeft | MouseButtonRight,
    MouseButtonMR = MouseButtonMiddle | MouseButtonRight,
    MouseButtonALL = MouseButtonLeft | MouseButtonMiddle | MouseButtonRight,
};

typedef struct  {
    float x; //鼠标位置的x坐标
    float y; //鼠标位置的y坐标
    ASPMouseAction action; //鼠标操作的action值，按下或松开。
    ASPMouseButtonMask button; //当前触发event的鼠标按键
    ASPMouseButtonMask buttonState; //鼠标的按键状态，可能存在其他按键已经被按下的情况
} ASPMouseButtonEvent;

typedef struct  {
    float x; //鼠标位置的x坐标
    float y; //鼠标位置的y坐标
    ASPMouseButtonMask buttonState; //鼠标操作的按键，左键，右键，中键
    bool forceShowCursor;
} ASPMouseMoveEvent;

/**
 Keyboard Operation
 */
typedef NS_ENUM(NSUInteger, ASPKeyboardAction) {
    KeyboardActionDown,
    KeyboardActionUp,
    KeyboardActionCancel,
};

typedef struct {
    NSInteger key; //操作的键值
    ASPKeyboardAction action;
} ASPKeyEvent;

typedef NS_ENUM(NSUInteger, AspNetworkQoS) {
    ASP_NETWORK_QOS_UNKOWN = 0,
    ASP_NETWORK_QOS_BEST = 1,
    ASP_NETWORK_QOS_GOOD = 2,
    ASP_NETWORK_QOS_AVERAGE = 3,
    ASP_NETWORK_QOS_BAD = 4,
};

typedef NS_ENUM(NSUInteger, AspEngineClientType) {
    ASP_ENGINE_CLIENT_DESKTOP = 0,
    ASP_ENGINE_CLIENT_CLOUDAPP = 1
};

// ASPConnParam
@interface ASPConnParam : NSObject
@property (nonatomic, copy) NSString* connIp;       //ip
@property (nonatomic, copy) NSString* connPort;     //port
@property (nonatomic, copy) NSString* connUdpPort;  //udp port
@property (nonatomic, copy) NSString* connToken;    //token for gateway
@property (nonatomic, copy) NSString *connTicket;
@property (nonatomic, copy) NSString* caFilePath;   //ca file for tls
@property (nonatomic, copy) NSString *desktopId;
@property (nonatomic, assign) BOOL enableTls;        //enable tls
@property (nonatomic, assign) BOOL enableStatistics; //enable profile statistics
@property (nonatomic, copy) NSString *user;
@property (nonatomic, assign) AspEngineClientType clientType;
@property (nonatomic, copy) NSString *uuid;
@property (nonatomic, assign) BOOL useVpc;           //use VPC
@end

// ASPConnTicket
@interface ASPConnTicket : NSObject
@property (nonatomic, copy) NSString* connTicket;    //connection ticket from server
@property (nonatomic, copy) NSString* caFilePath;    //ca file for tls
@property (nonatomic, copy) NSString *desktopId;
@property (nonatomic, assign) BOOL enableStatistics; //enable profile statistics
@property (nonatomic, assign) BOOL useVpc;           //use VPC
@property (nonatomic, copy) NSString *user;
@property (nonatomic, assign) AspEngineClientType clientType;
@property (nonatomic, copy) NSString *uuid;
@end

@interface ASPTouchEvent : NSObject
@property (nonatomic, assign) NSUInteger trackingId;
@property (nonatomic, assign) float x;
@property (nonatomic, assign) float y;
@property (nonatomic, assign) NSInteger touchType;
@property (nonatomic, assign) float pressure;
@end

//Log level
typedef NS_ENUM(NSUInteger, AspLogLevel) {
    AspLogLevelDebug,
    AspLogLevelInfo,
    AspLogLevelWarning,
    AspLogLevelError,
    AspLogLevelFatal,
};

typedef NS_ENUM(NSUInteger, AspShellSurfaceState) {
    ASP_SHELL_SURFACE_STATE_NORMAL,
    ASP_SHELL_SURFACE_STATE_MINIMIZED,
    ASP_SHELL_SURFACE_STATE_MAXIMIZED,
    ASP_SHELL_SURFACE_STATE_FULLSCREEN,
};

#pragma mark - ASPEngineDelegate
// ASPEngineDelegate
@protocol ASPEngineDelegate <NSObject>

- (void)onEngineErrorWithErrCode:(int)errCode
                          errMsg:(NSString*)errMsg;

- (void)onConnectionSuccess;

- (void)onConnectionFailureWithErrCode:(int)errCode
                                errMsg:(NSString*)errMsg;

- (void)onDisconnected:(int)reason;

- (void)onFirstFrameRendered:(long)timeCostMS;

- (void)onReconnect;

- (void)onPolicyUpdate:(NSString *)policy;

- (void)onUpdateNetworkQos:(AspNetworkQoS)qos;

- (void)onSessionSuccess;

- (void)onSuperResolutionStateChange:(BOOL)support;

@end

#pragma mark - ASPEngineResolutionUpdateDelegate
@protocol ASPEngineResolutionUpdateDelegate <NSObject>

- (void)onResolutionUpdateWithOldWidth:(int)oldWidth
                             oldHeight:(int)oldHeight
                                 width:(int)width
                                height:(int)height;

@end

#pragma mark - ASPEngineCursorDelegate
@protocol ASPEngineCursorDelegate <NSObject>
- (void)onCursorBitmapUpdateWithHotX:(int)hotX
                                hotY:(int)hotY
                               width:(int)width
                              height:(int)height
                                rgba:(char*)rgba;

- (void)onCursorReset;

- (void)onCursorHide;

- (void)onCursorMoveWithX:(int)x
                        y:(int)y;
@end

#pragma mark - ASPEngineOrientationUpdateDelegate
@protocol ASPEngineOrientationUpdateDelegate <NSObject>
- (void)onOrientationUpdateWithOldOrientation:(int)oldOrientation
                               newOrientation:(int)newOrientation;
@end

#pragma mark - ASPEngineStatisticsDelegate
@protocol ASPEngineStatisticsDelegate <NSObject>
- (void)onStatisticsInfoUpdate:(ASPStatisticsInfo *)info;
@end

#pragma mark - ASPEngineIMEDelegate
@protocol ASPEngineIMEDelegate <NSObject>
- (void)onIMEFocusUpdate:(bool)hasFocus;
- (void)onIMELocationUpdateWithX:(int)x
                               y:(int)y;
@end

#pragma mark - ASPEngineClipboardDelegate
@protocol ASPEngineClipboardDelegate <NSObject>
- (void)onClipBoardRead:(int)engineId type:(int *)type data:(uint8_t**)data size:(int *) size selection:(uint8_t)selection;
- (void)onClipBoardWrite:(int)engineId type:(int)type data:(uint8_t*)data size:(int)size;
@end

#pragma mark - ASPEngineGestureDelegate
@protocol ASPEngineGestureDelegate <NSObject>
- (void)onScale:(float)sx
             sy:(float)sy;
- (void)onTranslate:(float)tx
                 ty:(float)ty;
@end

#pragma mark - ASPEngineLogDelegate
@protocol ASPEngineLogDelegate <NSObject>
- (void)onLogMessage:(NSString*)msg
                 tag:(NSString*)tag
               level:(AspLogLevel)level;
@end

@protocol ASPWindowViewDelegate <NSObject>
- (void)onShellSurfaceActivate:(int32_t)windowId;
- (void)onShellSurfaceStateChange:(int32_t)windowId
                            state:(AspShellSurfaceState)state;
@end

#pragma mark - ASPEngineSDK
@interface ASPEngineSDK : NSObject

- (void)dispose;

- (void)enableRTC:(bool) enabled;

- (void)enableVDAagentCheck:(bool) enabled;

- (void)setDumpPath:(NSString *)path;

- (void)setNetworkDetectionConfig:(NSString *)json;

@property (nonatomic, assign) BOOL enableDump;
@property (nonatomic, assign) BOOL enableKeyBoardRepeat; // default YES

@property (nonatomic, weak) id<ASPEngineDelegate> engineDelegate;
@property (nonatomic, weak) id<ASPEngineResolutionUpdateDelegate> resolutionUpdateDelegate;
@property (nonatomic, weak) id<ASPEngineCursorDelegate> cursorDelegate;
@property (nonatomic, weak) id<ASPEngineOrientationUpdateDelegate> orientationUpdateDelegate;
@property (nonatomic, weak) id<ASPEngineStatisticsDelegate> statisticsDelegate;
@property (nonatomic, weak) id<ASPEngineIMEDelegate> imeDelegate;
@property (nonatomic, weak) id<ASPEngineClipboardDelegate> clipboardDelegate;
@property (nonatomic, weak) id<ASPEngineGestureDelegate> gestureDelegate;
@property (class, nonatomic, weak) id<ASPEngineLogDelegate> logDelegate;
@property (nonatomic, weak) id<ASPWindowViewDelegate> windowViewDelegate;

//start with ip port
- (void)start:(ASPConnParam*) connParam;

//start with ticket
- (void)startWithTicket:(ASPConnTicket*)ticketParam;

- (int)stop;

- (void)pause;

- (void)resume;

- (void)setRenderView:(UIView*) view;

- (void)setMaxSupportedVideoResolutionSize:(CGSize)size;

- (void)setVideoProfileWithWidth:(int)width
                          height:(int)height
                          fps:(int)fps; //视频流目标fps，目前不生效

- (void)sendTouchEvent:(ASPTouchEvent*) event
           touchAction:(ASPTouchAction)action;

- (void)sendTouchEvents:(NSArray<ASPTouchEvent*>*) eventArray
            touchAction:(ASPTouchAction)action
                  index:(int)index;

- (BOOL)sendKeyboardEvent: (ASPKeyEvent) event;

// Caps/Num/Scroll Lock
- (void)sendKeyboardLockModifiers: (int) modifers;

- (void)sendMouseButtonEvent: (ASPMouseButtonEvent)event;

- (void)sendMouseMoveEvent: (ASPMouseMoveEvent)event;

- (void)sendMouseWheelEvent:(int)delta
                buttonState:(ASPMouseButtonMask)buttonState;

- (void)setImePreedit: (NSString*)preeditStr;

- (void)setImeCommit: (NSString*)commitStr;

- (BOOL)enableMouseMode:(BOOL) enabled;

- (BOOL)enableStatistics:(BOOL) enabled;

- (void)enableDesktopMode:(BOOL)enabled;

- (void)enableDesktopGesture:(BOOL)enabled;

- (void)sendClipboardType:(int)type;

- (void)setSize:(CGSize)size;

- (void)enableSR:(BOOL) enabled;

@property (nonatomic, assign) BOOL mute;

- (void)showCursor;
- (void)hideCursor;

@end
