//
//  DesktopAddViewController.m
//  aspdemo
//
//  Created by 杨航涛 on 2022/1/12.
//

#import "DesktopAddViewController.h"

#import <ASPEngineSDK/ASPStreamView.h>
#include <objc/message.h>
#import <pthread.h>
#import "DebugToolManager.h"

@interface ASPDemoLog : NSObject<ASPEngineLogDelegate>

@property (nonatomic, strong) NSDateFormatter *logDateFormatter;

@end

@implementation ASPDemoLog

+ (instancetype)sharedInstance {
    static id obj = nil;
    static dispatch_once_t onceToken;
    dispatch_once(&onceToken, ^{
        obj = [self new];
    });
    return obj;
}

- (instancetype)init {
    self = [super init];
    if (self) {
        self.logDateFormatter = [[NSDateFormatter alloc] init];
        [self.logDateFormatter setDateFormat:@"yyyy-MM-dd HH:mm:ss.SSS"];
    }
    return self;
}

- (void)onLogMessage:(NSString *)msg tag:(NSString *)tag level:(AspLogLevel)level {
    NSString* LevelStr =
    level == AspLogLevelDebug ? @"DEBUG" :
    level == AspLogLevelInfo? @"INFO" :
    level == AspLogLevelWarning ? @"WARN" :
    level == AspLogLevelError ? @"ERROR" :
    level == AspLogLevelFatal ? @"FATAL" : @"DEBUG";
    __uint64_t tid;
    pthread_threadid_np(NULL, &tid);
    NSString* finalMsg = [NSString stringWithFormat:@"%@ %@ [%d:%llu] %@.%@", [self.logDateFormatter stringFromDate:[NSDate date]], LevelStr, getpid(),tid, tag, msg];
    printf("%s\n", [finalMsg cStringUsingEncoding:NSUTF8StringEncoding]);
}

@end

@interface DesktopAddViewController () <ASPEngineDelegate, DebugToolManagerDelegate, ASPEngineStatisticsDelegate, ASPWindowViewDelegate>

@property (nonatomic, strong) ASPStreamView *streamView;
@property (nonatomic, strong) DebugToolManager *DTMObj;
@property (nonatomic, assign) BOOL enableMouseMode;
@property (nonatomic, assign) BOOL supportSR;

@end

@implementation DesktopAddViewController

- (void)viewDidLoad {
    [super viewDidLoad];
    self.view.multipleTouchEnabled = YES;

    _enableMouseMode = NO;
    [ASPStreamView setASPEngineLogDelegate:[ASPDemoLog sharedInstance]];
    
    ASPStreamView* streamView = [ASPStreamView buildStreamView];
    [streamView setEngineDelegate:self];
    [streamView setWindowViewDelegate:self];
    if (self.connParam.connTicket) {
        ASPConnTicket* ticketParam = [[ASPConnTicket alloc] init];
        ticketParam.connTicket = self.connParam.connTicket;
        ticketParam.useVpc = false;
        ticketParam.enableStatistics = true;
        ticketParam.desktopId = self.connParam.desktopId;
        [streamView startWithTicket:ticketParam];
    } else {
        ASPConnParam* connParam = [[ASPConnParam alloc] init];
        connParam.connIp = self.connParam.connIp;
        connParam.connPort = self.connParam.connPort;
        connParam.connUdpPort = self.connParam.connUdpPort;
        connParam.connToken = self.connParam.connToken;
        connParam.enableTls = self.connParam.enableTls;
        connParam.desktopId = self.connParam.desktopId;
        connParam.enableStatistics = true;
        [streamView start:connParam];
    }
    self.streamView = streamView;

    [self.view addSubview:streamView];
    
    [self.streamView setStatisticsDelegate:self];
  
    self.DTMObj = [[DebugToolManager alloc] init];
    self.DTMObj.delegate =  self;
    [self.DTMObj loadDebugView:self.view streamView:self.streamView];
}

- (void)onSetCursorShow:(BOOL)show {
    show ? [self.streamView showCursor] : [self.streamView hideCursor];
}

- (void)onConnectionSuccess {
    NSLog(@"[ASPEngineDelegate] onConnectionSuccess");
    [self.streamView enableStatistics:true];
    [self.DTMObj hideLoading];
}

- (void)onDisconnected:(int)reason {
    NSLog(@"[ASPEngineDelegate] onDisconnected");
    [self dismissViewController];
}

// TODO not support yet
- (void)onReconnect {
    NSLog(@"[ASPEngineDelegate] onReconnect");
    [self.DTMObj showLoading:@"正在重连......"];
}

- (void)onConnectionFailureWithErrCode:(int)errCode errMsg:(NSString *)errMsg {
    NSLog(@"[ASPEngineDelegate] onConnectionFailureWithErrCode:%d, errMsg:%@", errCode, errMsg);
    [self dismissViewController];
}

- (void)onEngineErrorWithErrCode:(int)errCode errMsg:(NSString *)errMsg {
    NSLog(@"[ASPEngineDelegate] onEngineErrorWithErrCode:%d, errMsg:%@", errCode, errMsg);
}

- (void)onFirstFrameRendered:(long)timeCostMS {
    NSLog(@"[ASPEngineDelegate] onFirstFrameRendered");
}

- (void)onPolicyUpdate:(NSString *)policy {
    NSLog(@"[ASPEngineDelegate] onPolicyUpdate:%@", policy);
}

- (void)onUpdateNetworkQos:(AspNetworkQoS)qos {
    NSLog(@"[ASPEngineDelegate] onUpdateNetWorkQos:%@", @(qos));
    if (qos == ASP_NETWORK_QOS_BAD) {
        [self.DTMObj showLoading:@"当前弱网络环境"];
    } else {
        [self.DTMObj hideLoading];
    }
}

- (void)onSuperResolutionStateChange:(BOOL)support {
    NSLog(@"[ASPEngineDelegate] onSuperResolutionStateChange:%@", @(support));
    self.supportSR = support;
}

#pragma mark - ASPWindowViewDelegate
- (void)onShellSurfaceActivate:(int32_t)windowId {
    NSLog(@"[ASPWindowViewDelegate] onActivate:%@", @(windowId));
}

- (void)onShellSurfaceStateChange:(int32_t)windowId
                            state:(AspShellSurfaceState)state {
    NSLog(@"[ASPWindowViewDelegate] onChange:%@ state:%@", @(windowId), @(state));
    if (state == ASP_SHELL_SURFACE_STATE_MINIMIZED) {
        [self dismissViewController];
    }
}

#pragma mark - DebugToolManagerDelegate

- (void)onSendKeyboardEvent:(ASPKeyEvent)event {
    [self.streamView sendKeyboardEvent:event];
}

- (void)onChangeResolutionRatioListAction {
    NSLog(@"onChangeResolutionRatioListAction");
    [self showResolutionRatioList];
}

- (void)onTouchFeelAction {
    UIAlertController *alertController = [UIAlertController alertControllerWithTitle:@"选择" message:nil preferredStyle:UIAlertControllerStyleActionSheet];
    alertController.popoverPresentationController.sourceView = self.view;
    alertController.popoverPresentationController.sourceRect = CGRectMake((self.view.bounds.size.width-100)/2, self.view.bounds.size.height - 100, 100, 100);
    UIAlertAction *action1 = [UIAlertAction actionWithTitle:@"开启" style:UIAlertActionStyleDefault handler:^(UIAlertAction * _Nonnull action) {
        self.streamView.enableTouchFeel = YES;
    }];
    UIAlertAction *action2 = [UIAlertAction actionWithTitle:@"关闭" style:UIAlertActionStyleDefault handler:^(UIAlertAction * _Nonnull action) {
        self.streamView.enableTouchFeel = NO;
    }];
    UIAlertAction *action4 = [UIAlertAction actionWithTitle:@"取消" style:UIAlertActionStyleCancel handler:^(UIAlertAction * _Nonnull action) {
        
    }];
    [alertController addAction:action1];
    [alertController addAction:action2];
    [alertController addAction:action4];
    [self presentViewController:alertController animated:YES completion:nil];
}

- (void)doSuperResolutionAction {
    if (!self.supportSR) {
        [self.DTMObj showLoading:@"不支持超分"];
        [self.DTMObj performSelector:@selector(hideLoading) withObject:nil afterDelay:3];
        return;
    }
    UIAlertController *alertController = [UIAlertController alertControllerWithTitle:@"选择" message:nil preferredStyle:UIAlertControllerStyleActionSheet];
    alertController.popoverPresentationController.sourceView = self.view;
    alertController.popoverPresentationController.sourceRect = CGRectMake((self.view.bounds.size.width-100)/2, self.view.bounds.size.height - 100, 100, 100);
    UIAlertAction *action1 = [UIAlertAction actionWithTitle:@"开启" style:UIAlertActionStyleDefault handler:^(UIAlertAction * _Nonnull action) {
        [self.streamView enableSR:TRUE];
    }];
    UIAlertAction *action2 = [UIAlertAction actionWithTitle:@"关闭" style:UIAlertActionStyleDefault handler:^(UIAlertAction * _Nonnull action) {
        [self.streamView enableSR:FALSE];
    }];
    UIAlertAction *action4 = [UIAlertAction actionWithTitle:@"取消" style:UIAlertActionStyleCancel handler:^(UIAlertAction * _Nonnull action) {
        
    }];
    [alertController addAction:action1];
    [alertController addAction:action2];
    [alertController addAction:action4];
    [self presentViewController:alertController animated:YES completion:nil];
}

- (void)doGestureAction {
    UIAlertController *alertController = [UIAlertController alertControllerWithTitle:@"选择" message:nil preferredStyle:UIAlertControllerStyleActionSheet];
    alertController.popoverPresentationController.sourceView = self.view;
    alertController.popoverPresentationController.sourceRect = CGRectMake((self.view.bounds.size.width-100)/2, self.view.bounds.size.height - 100, 100, 100);
    UIAlertAction *action1 = [UIAlertAction actionWithTitle:@"开启" style:UIAlertActionStyleDefault handler:^(UIAlertAction * _Nonnull action) {
        [self.streamView enableDesktopGesture:YES];
    }];
    UIAlertAction *action2 = [UIAlertAction actionWithTitle:@"关闭" style:UIAlertActionStyleDefault handler:^(UIAlertAction * _Nonnull action) {
        [self.streamView enableDesktopGesture:NO];
    }];
    UIAlertAction *action4 = [UIAlertAction actionWithTitle:@"取消" style:UIAlertActionStyleCancel handler:^(UIAlertAction * _Nonnull action) {
        
    }];
    [alertController addAction:action1];
    [alertController addAction:action2];
    [alertController addAction:action4];
    [self presentViewController:alertController animated:YES completion:nil];
}

- (void)doTouchAction {
    UIAlertController *alertController = [UIAlertController alertControllerWithTitle:@"选择" message:nil preferredStyle:UIAlertControllerStyleActionSheet];
    alertController.popoverPresentationController.sourceView = self.view;
    alertController.popoverPresentationController.sourceRect = CGRectMake((self.view.bounds.size.width-100)/2, self.view.bounds.size.height - 100, 100, 100);
    UIAlertAction *action1 = [UIAlertAction actionWithTitle:@"开启云端touch" style:UIAlertActionStyleDefault handler:^(UIAlertAction * _Nonnull action) {
        [self.streamView enableDesktopMode:NO];
    }];
    UIAlertAction *action2 = [UIAlertAction actionWithTitle:@"关闭云端touch" style:UIAlertActionStyleDefault handler:^(UIAlertAction * _Nonnull action) {
        [self.streamView enableDesktopMode:YES];
    }];
    UIAlertAction *action4 = [UIAlertAction actionWithTitle:@"取消" style:UIAlertActionStyleCancel handler:^(UIAlertAction * _Nonnull action) {
        
    }];
    [alertController addAction:action1];
    [alertController addAction:action2];
    [alertController addAction:action4];
    [self presentViewController:alertController animated:YES completion:nil];
}

- (void)onMuteAction {
    UIAlertController *alertController = [UIAlertController alertControllerWithTitle:@"选择" message:nil preferredStyle:UIAlertControllerStyleActionSheet];
    alertController.popoverPresentationController.sourceView = self.view;
    alertController.popoverPresentationController.sourceRect = CGRectMake((self.view.bounds.size.width-100)/2, self.view.bounds.size.height - 100, 100, 100);
    UIAlertAction *action1 = [UIAlertAction actionWithTitle:@"开启静音" style:UIAlertActionStyleDefault handler:^(UIAlertAction * _Nonnull action) {
        self.streamView.mute = YES;
    }];
    UIAlertAction *action2 = [UIAlertAction actionWithTitle:@"关闭静音" style:UIAlertActionStyleDefault handler:^(UIAlertAction * _Nonnull action) {
        self.streamView.mute = NO;
    }];
    UIAlertAction *action4 = [UIAlertAction actionWithTitle:@"取消" style:UIAlertActionStyleCancel handler:^(UIAlertAction * _Nonnull action) {
        
    }];
    [alertController addAction:action1];
    [alertController addAction:action2];
    [alertController addAction:action4];
    [self presentViewController:alertController animated:YES completion:nil];
}

- (void)showResolutionRatioList {
    UIAlertController *alertController = [UIAlertController alertControllerWithTitle:@"选择" message:nil preferredStyle:UIAlertControllerStyleActionSheet];
    alertController.popoverPresentationController.sourceView = self.view;
    alertController.popoverPresentationController.sourceRect = CGRectMake((self.view.bounds.size.width-100)/2, self.view.bounds.size.height - 100, 100, 100);
    UIAlertAction *action1 = [UIAlertAction actionWithTitle:@"2224*1668" style:UIAlertActionStyleDefault handler:^(UIAlertAction * _Nonnull action) {
        [self.streamView setVideoProfileWithWidth:2224 height:1668 fps:30];
    }];
    UIAlertAction *action2 = [UIAlertAction actionWithTitle:@"1668*1251" style:UIAlertActionStyleDefault handler:^(UIAlertAction * _Nonnull action) {
        [self.streamView setVideoProfileWithWidth:1668 height:1251 fps:30];
    }];
    UIAlertAction *action3 = [UIAlertAction actionWithTitle:@"1112*834" style:UIAlertActionStyleDefault handler:^(UIAlertAction * _Nonnull action) {
        [self.streamView setVideoProfileWithWidth:1112 height:834 fps:30];
    }];
    UIAlertAction *action4 = [UIAlertAction actionWithTitle:@"取消" style:UIAlertActionStyleCancel handler:^(UIAlertAction * _Nonnull action) {
        
    }];
    [alertController addAction:action1];
    [alertController addAction:action2];
    [alertController addAction:action3];
    [alertController addAction:action4];
    [self presentViewController:alertController animated:YES completion:nil];
}

- (void)onClickSimulateMouse {
    NSLog(@"onClickSimulateMouse");
    [self.streamView simulateMouseClick:YES]; //left click
    //[self.streamView simulateMouseClick:NO]; //right click
}

- (void)onEnableMouseModeAction {
    if (_enableMouseMode == NO) {
        _enableMouseMode = YES;
    } else {
        _enableMouseMode = NO;
    }
    NSLog(@"streamView enableMouseMode:%d", _enableMouseMode);
    [self.streamView enableMouseMode:_enableMouseMode];
}

- (void)onDisconnectingAction {
    [self dismissViewController];
}

- (void)dismissViewController {
    [self.DTMObj hideLoading];
    self.streamView.statisticsDelegate = nil;
    [self.streamView dispose];
    __weak typeof(self) weakSelf = self;
    dispatch_async(dispatch_get_main_queue(), ^{
        [weakSelf dismissViewControllerAnimated:YES completion:^{
//            [[NSNotificationCenter defaultCenter] postNotificationName:@"openTestAction" object:nil];
        }];
    });
}

- (NSString*) getTokenFromTicket:(NSString*) ticket {
    NSData* data = [[NSData alloc]initWithBase64EncodedString:ticket options:0];
    NSString* decodeStr = [[NSString alloc]initWithData:data encoding:NSUTF8StringEncoding];
    NSLog(@"parseTicket decodeStr:%@", decodeStr);
    
    NSString *gwToken = nil;
    NSArray* array = [decodeStr componentsSeparatedByString:@"\r\n"];
    for (NSString* str in array) {
        NSRange GWTokenRange = [str rangeOfString:@"GWToken="];
        NSRange GatewayAddressRange = [str rangeOfString:@"GatewayAddress="];
        NSRange GatewayPortRange = [str rangeOfString:@"GatewayPort="];
        NSRange VPCEndpoint = [str rangeOfString:@"VPCEndpoint="];
        if (GWTokenRange.location != NSNotFound) {
            gwToken = [str substringFromIndex:(GWTokenRange.length)];
        } else if (GatewayAddressRange.location != NSNotFound) {
//            gwInfo.gatewayAddress = [str substringFromIndex:(GatewayAddressRange.length)];
        } else if (GatewayPortRange.location != NSNotFound) {
//            gwInfo.gatewayPort = [str substringFromIndex:(GatewayPortRange.length)];
        } else if (VPCEndpoint.location != NSNotFound) {
//            gwInfo.vpcEndpoint = [str substringFromIndex:(VPCEndpoint.length)];
        }
    }
    return gwToken;
}

#pragma mark - UIViewController

- (BOOL)prefersStatusBarHidden {
    return YES;
}

- (void)onStatisticsInfoUpdate:(ASPStatisticsInfo *)info {
    NSLog(@"onStatisticsInfoUpdate, networkLatency:%ld", info.mNetworkLatencyMS);
}


- (BOOL)prefersHomeIndicatorAutoHidden {
    return YES;
}

@end
